/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private define ------------------------------------------------------------*/
#define BOARD_IN_MASTER_MODE
#define DATA_LENGTH       16

/* Private variables ---------------------------------------------------------*/
I2S_HandleTypeDef I2sHandle;
uint16_t TxBuff[16] = {1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16};
uint16_t RxBuff[16] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
static void APP_WaitAndCheckEndOfTransfer(void);
static uint8_t APP_Buffercmp8(uint8_t* pBuffer1, uint8_t* pBuffer2, uint8_t BufferLength);
static void APP_LedBlinking(void);

/**
  * @brief  Main program
  * @retval int
  */
int main(void)
{
  /* Reset of all peripherals, Initializes the Systick */
  HAL_Init();
  
  /* Configure Systemclock */
  APP_SystemClockConfig(); 
  
  /* Initialize LED */
  BSP_LED_Init(LED_GREEN);

  /* Initialization button */
  BSP_PB_Init(BUTTON_KEY,BUTTON_MODE_GPIO);

  /* Wait for the button to be pressed */
  while(BSP_PB_GetState(BUTTON_KEY) == 0);

  /* Initialize I2S */
  I2sHandle.Instance          = SPI1;
#ifdef BOARD_IN_MASTER_MODE
  I2sHandle.Init.Mode         = I2S_MODE_MASTER_TX;  /* Master transmit */
#else
  I2sHandle.Init.Mode         = I2S_MODE_SLAVE_RX;   /* Slave receive */
#endif
  I2sHandle.Init.Standard     = I2S_STANDARD_PHILIPS;
  I2sHandle.Init.DataFormat   = I2S_DATAFORMAT_16B;
  I2sHandle.Init.MCLKOutput   = I2S_MCLKOUTPUT_DISABLE;
  I2sHandle.Init.AudioFreq    = I2S_AUDIOFREQ_8K;
  I2sHandle.Init.CPOL         = I2S_CPOL_HIGH;
  /* Initialize SPI peripheral */
  if (HAL_I2S_Init(&I2sHandle) != HAL_OK)
  {
    APP_ErrorHandler();
  }

#ifdef BOARD_IN_MASTER_MODE
  /* Transmit an amount of data in non-blocking mode with interrupt */
  if (HAL_I2S_Transmit_IT(&I2sHandle, (uint16_t *)TxBuff, DATA_LENGTH) != HAL_OK)
  {
    APP_ErrorHandler();
  }
#else
  /* Receive an amount of data in non-blocking mode with interrupt */
  if (HAL_I2S_Receive_IT(&I2sHandle, (uint16_t *)RxBuff, DATA_LENGTH) != HAL_OK)
  {
    APP_ErrorHandler();
  }
#endif
  /* Wait for transmit completion */
  while (HAL_I2S_GetState(&I2sHandle) != HAL_I2S_STATE_READY)
  {
  }

#ifdef BOARD_IN_MASTER_MODE
  HAL_Delay(5); /* Wait for the slave to be ready for data transmission */
#endif

  /* Initialize I2S */
  I2sHandle.Instance          = SPI1;
#ifdef BOARD_IN_MASTER_MODE
  I2sHandle.Init.Mode         = I2S_MODE_MASTER_RX;  /* Master receive */
#else
  I2sHandle.Init.Mode         = I2S_MODE_SLAVE_TX;   /* Slave transmit */
#endif
  I2sHandle.Init.Standard     = I2S_STANDARD_PHILIPS;
  I2sHandle.Init.DataFormat   = I2S_DATAFORMAT_16B;
  I2sHandle.Init.MCLKOutput   = I2S_MCLKOUTPUT_DISABLE;
  I2sHandle.Init.AudioFreq    = I2S_AUDIOFREQ_8K;
  I2sHandle.Init.CPOL         = I2S_CPOL_HIGH;  
  if (HAL_I2S_Init(&I2sHandle) != HAL_OK)
  {
    APP_ErrorHandler();
  }

#ifdef BOARD_IN_MASTER_MODE
  /* Receive an amount of data in non-blocking mode with interrupt */
  if (HAL_I2S_Receive_IT(&I2sHandle, (uint16_t *)RxBuff, DATA_LENGTH) != HAL_OK)
  {
    APP_ErrorHandler();
  }
#else
  /* Transmit data as slave */
  if (HAL_I2S_Transmit_IT(&I2sHandle, (uint16_t *)TxBuff, DATA_LENGTH) != HAL_OK)
  {
    APP_ErrorHandler();
  }
#endif
  /* Wait for receive completion  */
  while (HAL_I2S_GetState(&I2sHandle) != HAL_I2S_STATE_READY)
  {
  }

  /* Wait for the transfer be completed and check the received data */
  APP_WaitAndCheckEndOfTransfer();
  
  while (1)
  {
  }
}

/**
  * @brief  Wait for the transfer be completed and check the received data
  * @param  None
  * @retval None
  */
static void APP_WaitAndCheckEndOfTransfer(void)
{
  /* Wait for the transfer be completed */
  while (I2sHandle.State != HAL_I2S_STATE_READY)
  {}

  /* Compare sent and received data */
  if(APP_Buffercmp8((uint8_t*)TxBuff, (uint8_t*)RxBuff, DATA_LENGTH))
  {
    /* error handling */
    APP_LedBlinking();
  }
  else
  {
    /* If data is received, the LED is turned on */
    BSP_LED_On(LED_GREEN);
  }
}

/**
  * @brief  Character comparison function
  * @param  pBuffer1: Pointer to buffer 1 that to be compared 
  * @param  pBuffer2: Pointer to buffer 2 that to be compared 
  * @param  BufferLength: The number of characters to be compared
  * @retval 0: The comparison value is the same; 1: The comparison value is different
  */
static uint8_t APP_Buffercmp8(uint8_t* pBuffer1, uint8_t* pBuffer2, uint8_t BufferLength)
{
  while (BufferLength--)
  {
    if (*pBuffer1 != *pBuffer2)
    {
      return 1;
    }
    pBuffer1++;
    pBuffer2++;
  }

  return 0;
}

/**
  * @brief  LED blink
  * @param  None
  * @retval None
  */
static void APP_LedBlinking(void)
{
  while (1)
  {
    BSP_LED_Toggle(LED_GREEN); 
    HAL_Delay(500);
  }
}

/**
  * @brief  System clock configuration function
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

  /* Oscillator configuration */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_LSI | RCC_OSCILLATORTYPE_MSI; /* Select oscillator HSI, LSI, MSI */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType |= RCC_OSCILLATORTYPE_HSE; /* Select oscillator HSE */
#endif /* RCC_HSE_SUPPORT */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType |= RCC_OSCILLATORTYPE_LSE; /* Select oscillator LSE */
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.HSIState = RCC_HSI_ON;                          /* Enable HSI */
  RCC_OscInitStruct.HSIDiv = RCC_HSI_DIV1;                          /* HSI 1 frequency division */
  RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_8MHz;  /* Configure HSI clock 8MHz */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.HSEState = RCC_HSE_OFF;                         /* Close HSE */
  /*RCC_OscInitStruct.HSEFreq = RCC_HSE_24_32MHz;*/
#endif /* RCC_HSE_SUPPORT */
  RCC_OscInitStruct.LSIState = RCC_LSI_OFF;                         /* Close LSI */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.LSEState = RCC_LSE_OFF;                         /* Close LSE */
  /*RCC_OscInitStruct.LSEDriver = RCC_LSEDRIVE_MEDIUM;*/
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.MSIState = RCC_MSI_OFF;                         /* Close MSI */
  /*RCC_OscInitStruct.MSIDiv = RCC_MSI_DIV1;*/
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_OFF;                     /* Close PLL */
  /*RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSI;*/
  /*RCC_OscInitStruct.PLL.PLLMUL = RCC_PLL_MUL2;*/
  /* Configure oscillator */
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    APP_ErrorHandler();
  }

  /* Clock source configuration */
  RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_PCLK; /* Choose to configure clock HCLK, SYSCLK, PCLK */
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_HSISYS; /* Select HSISYS as the system clock */
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;        /* AHB clock 1 division */
  RCC_ClkInitStruct.APBCLKDivider = RCC_HCLK_DIV1;          /* APB clock 1 division */
  /* Configure clock source */
  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_0) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
