/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private define ------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
FDCAN_HandleTypeDef FdcanHandle;
uint8_t TxData[64] = {0};
FDCAN_FilterTypeDef FilterConfig;
FDCAN_RxHeaderTypeDef RxHeader;
uint8_t RxData[64] = {0};
FDCAN_TxHeaderTypeDef TxHeader;
uint8_t DLCtoBytes[] = {0, 1, 2, 3, 4, 5, 6, 7, 8, 12, 16, 20, 24, 32, 48, 64};

/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);

/**
  * @brief  Main program.
  * @retval int
  */
int main(void)
{
  uint32_t i = 0;
  
  /* Reset of all peripherals, Initializes the Systick. */
  HAL_Init();
  
  /* Config Voltage of MR */
  HAL_PWREx_SetMrVoltageMode(PWR_MR_VOLTAGE_OVERCLOCK);
  
  /* Configure system clock */
  APP_SystemClockConfig();

  /* Initialize debug UART (used for printf) */
  BSP_UART_Config();
  
  /* CAN init */
  FdcanHandle.Instance = FDCAN;
  FdcanHandle.Init.FrameFormat = FDCAN_FRAME_FD_BRS;
  FdcanHandle.Init.Mode = FDCAN_MODE_NORMAL;
  FdcanHandle.Init.AutoRetransmission = ENABLE;
  FdcanHandle.Init.TransmitPause = DISABLE;
  FdcanHandle.Init.ProtocolException = ENABLE;
  FdcanHandle.Init.NominalPrescaler = 1;   /* 500kbps */
  FdcanHandle.Init.NominalSyncJumpWidth = 10;
  FdcanHandle.Init.NominalTimeSeg1 = 37;
  FdcanHandle.Init.NominalTimeSeg2 = 10;
  FdcanHandle.Init.DataPrescaler = 1;   /* 2Mbps */
  FdcanHandle.Init.DataSyncJumpWidth = 3;
  FdcanHandle.Init.DataTimeSeg1 = 8;
  FdcanHandle.Init.DataTimeSeg2 = 3;
  FdcanHandle.Init.MessageRAMOffset = 0;
  FdcanHandle.Init.StdFiltersNbr = 0;
  FdcanHandle.Init.ExtFiltersNbr = 1;
  FdcanHandle.Init.RxFifo0ElmtsNbr = 1;
  FdcanHandle.Init.RxFifo0ElmtSize = FDCAN_DATA_BYTES_64;
  FdcanHandle.Init.RxFifo1ElmtsNbr = 0;
  FdcanHandle.Init.RxBuffersNbr = 0;
  FdcanHandle.Init.TxEventsNbr = 0;
  FdcanHandle.Init.TxBuffersNbr = 0;
  FdcanHandle.Init.TxFifoQueueElmtsNbr = 1;
  FdcanHandle.Init.TxFifoQueueMode = FDCAN_TX_FIFO_OPERATION;
  FdcanHandle.Init.TxElmtSize = FDCAN_DATA_BYTES_64;
  HAL_FDCAN_Init(&FdcanHandle);
  
  /* Configure Rx filter */
  FilterConfig.IdType = FDCAN_EXTENDED_ID;
  FilterConfig.FilterIndex = 0;
  FilterConfig.FilterType = FDCAN_FILTER_MASK;
  FilterConfig.FilterConfig = FDCAN_FILTER_TO_RXFIFO0;
  FilterConfig.FilterID1 = 0x12345678;
  FilterConfig.FilterID2 = 0x1FFFFFFF; /* For acceptance, MessageID and FilterID1 must match exactly */
  HAL_FDCAN_ConfigFilter(&FdcanHandle, &FilterConfig);
  
  /* Configure global filter to reject all non-matching frames */
  HAL_FDCAN_ConfigGlobalFilter(&FdcanHandle, FDCAN_REJECT, FDCAN_REJECT, FDCAN_REJECT_REMOTE, FDCAN_REJECT_REMOTE);
  
  /* Prepare Tx Header */
  TxHeader.Identifier = 0x12345678;
  TxHeader.IdType = FDCAN_EXTENDED_ID;
  TxHeader.TxFrameType = FDCAN_DATA_FRAME;
  TxHeader.DataLength = FDCAN_DLC_BYTES_64;
  TxHeader.ErrorStateIndicator = FDCAN_ESI_ACTIVE;
  TxHeader.BitRateSwitch = FDCAN_BRS_ON;
  TxHeader.FDFormat = FDCAN_FD_CAN;
  TxHeader.TxEventFifoControl = FDCAN_NO_TX_EVENTS;
  TxHeader.MessageMarker = 0;
  
  /* Sending data initialization */
  for (i = 0; i <64; i++)
  {
    TxData[i] = i;
  }
  
  /* Start the FDCAN module */
  HAL_FDCAN_Start(&FdcanHandle);

  /* Add messages to TX FIFO */
  HAL_FDCAN_AddMessageToTxFifoQ(&FdcanHandle, &TxHeader, &TxData[0]);
  
  /* Infinite loop */
  while (1)
  {
    /* Wait for two messages received */
    while (HAL_FDCAN_GetRxFifoFillLevel(&FdcanHandle, FDCAN_RX_FIFO0) < 1) {}
    
    /* Retrieve Rx messages from RX FIFO0 */
    HAL_FDCAN_GetRxMessage(&FdcanHandle, FDCAN_RX_FIFO0, &RxHeader, RxData);
    
    /* Print received data */
    for (i = 0; i < DLCtoBytes[RxHeader.DataLength >> 16]; i++)
    {
      printf("%x ", RxData[i]);
    }
  }
}

/**
  * @brief  System clock configuration function
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

  /* Oscillator configuration */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_LSI | RCC_OSCILLATORTYPE_MSI; /* Select oscillator HSI, LSI, MSI */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType |= RCC_OSCILLATORTYPE_HSE; /* Select oscillator HSE */
#endif /* RCC_HSE_SUPPORT */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType |= RCC_OSCILLATORTYPE_LSE; /* Select oscillator LSE */
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.HSIState = RCC_HSI_ON;                          /* Enable HSI */
  RCC_OscInitStruct.HSIDiv = RCC_HSI_DIV1;                          /* HSI 1 frequency division */
  RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_24MHz;  /* Configure HSI clock 24MHz */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.HSEState = RCC_HSE_OFF;                         /* Close HSE */
  /*RCC_OscInitStruct.HSEFreq = RCC_HSE_24_32MHz;*/
#endif /* RCC_HSE_SUPPORT */
  RCC_OscInitStruct.LSIState = RCC_LSI_OFF;                         /* Close LSI */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.LSEState = RCC_LSE_OFF;                         /* Close LSE */
  /*RCC_OscInitStruct.LSEDriver = RCC_LSEDRIVE_MEDIUM;*/
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.MSIState = RCC_MSI_OFF;                         /* Close MSI */
  /*RCC_OscInitStruct.MSIDiv = RCC_MSI_DIV1;*/
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_OFF;                     /* Close PLL */
  /*RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSI;*/
  /*RCC_OscInitStruct.PLL.PLLMUL = RCC_PLL_MUL2;*/
  /* Configure oscillator */
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    APP_ErrorHandler();
  }

  /* Clock source configuration */
  RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_PCLK; /* Choose to configure clock HCLK, SYSCLK, PCLK */
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_HSISYS; /* Select HSISYS as the system clock */
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;        /* AHB clock 1 division */
  RCC_ClkInitStruct.APBCLKDivider = RCC_HCLK_DIV1;          /* APB clock 1 division */
  /* Configure clock source */
  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_0) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
