/**
  ******************************************************************************
  * @file    py32t092_ll_rcc.h
  * @author  MCU Application Team
  * @brief   Header file of RCC LL module.
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __PY32T092_LL_RCC_H
#define __PY32T092_LL_RCC_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "py32t0xx.h"

/** @addtogroup PY32T092_LL_Driver
  * @{
  */

#if defined(RCC)

/** @defgroup RCC_LL RCC
  * @{
  */

/* Private types -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/** @defgroup RCC_LL_Private_Variables RCC Private Variables
  * @{
  */


/**
  * @}
  */

/* Private constants ---------------------------------------------------------*/
/* Private macros ------------------------------------------------------------*/
#if defined(USE_FULL_LL_DRIVER)
/** @defgroup RCC_LL_Private_Macros RCC Private Macros
  * @{
  */
/**
  * @}
  */
#endif /*USE_FULL_LL_DRIVER*/

/* Exported types ------------------------------------------------------------*/
#if defined(USE_FULL_LL_DRIVER)
/** @defgroup RCC_LL_Exported_Types RCC Exported Types
  * @{
  */

/** @defgroup LL_ES_CLOCK_FREQ Clocks Frequency Structure
  * @{
  */

/**
  * @brief  RCC Clocks Frequency Structure
  */
typedef struct
{
  uint32_t SYSCLK_Frequency;        /*!< SYSCLK clock frequency */
  uint32_t HCLK_Frequency;          /*!< HCLK clock frequency */
  uint32_t PCLK1_Frequency;         /*!< PCLK1 clock frequency */
} LL_RCC_ClocksTypeDef;

/**
  * @}
  */

/**
  * @}
  */
#endif /* USE_FULL_LL_DRIVER */

/* Exported constants --------------------------------------------------------*/
/** @defgroup RCC_LL_Exported_Constants RCC Exported Constants
  * @{
  */

/** @defgroup RCC_LL_EC_OSC_VALUES Oscillator Values adaptation
  * @brief    Defines used to adapt values of different oscillators
  * @note     These values could be modified in the user environment according to
  *           HW set-up.
  * @{
  */
#if defined(RCC_HSE_SUPPORT)
#if !defined  (HSE_VALUE)
#define HSE_VALUE    24000000U   /*!< Value of the HSE oscillator in Hz */
#endif /* HSE_VALUE */
#endif

#if !defined  (HSI_VALUE)
#define HSI_VALUE    8000000U  /*!< Value of the HSI oscillator in Hz */
#endif /* HSI_VALUE */

#if defined(RCC_LSE_SUPPORT)
#if !defined  (LSE_VALUE)
#define LSE_VALUE    32768U     /*!< Value of the LSE oscillator in Hz */
#endif /* LSE_VALUE */
#endif

#if !defined  (LSI_VALUE)
#define LSI_VALUE    32768U     /*!< Value of the LSI oscillator in Hz */
#endif /* LSI_VALUE */

#if !defined  (MSI_VALUE)
#define MSI_VALUE    2000000UL /*!< Value of the Internal oscillator in Hz*/
#endif /* MSI_VALUE */

/**
  * @}
  */

/** @defgroup RCC_LL_EC_CLEAR_FLAG Clear Flags Defines
  * @brief    Flags defines which can be used with LL_RCC_WriteReg function
  * @{
  */
#define LL_RCC_CICR_LSIRDYC                RCC_CICR_LSIRDYC     /*!< LSI Ready Interrupt Clear */
#define LL_RCC_CICR_HSIRDYC                RCC_CICR_HSIRDYC     /*!< HSI Ready Interrupt Clear */
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_CICR_HSERDYC                RCC_CICR_HSERDYC     /*!< HSE Ready Interrupt Clear */
#endif
#if defined(RCC_PLL_SUPPORT)
#define LL_RCC_CICR_PLLRDYC                RCC_CICR_PLLRDYC     /*!< PLL Ready Interrupt Clear */
#endif
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_CICR_LSERDYC                RCC_CICR_LSERDYC     /*!< LSE Ready Interrupt Clear */
#define LL_RCC_CICR_LSECSSC                RCC_CICR_LSECSSC     /*!< LSE Clock Security System Interrupt Clear */
#endif
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_CICR_CSSC                   RCC_CICR_CSSC        /*!< Clock Security System Interrupt Clear */
#endif
#define LL_RCC_CICR_MSIRDYC                RCC_CICR_MSIRDYC     /*!< MSI Ready Interrupt Clear */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_GET_FLAG Get Flags Defines
  * @brief    Flags defines which can be used with LL_RCC_ReadReg function
  * @{
  */
#define LL_RCC_CIFR_LSIRDYF                RCC_CIFR_LSIRDYF     /*!< LSI Ready Interrupt flag */
#define LL_RCC_CIFR_HSIRDYF                RCC_CIFR_HSIRDYF     /*!< HSI Ready Interrupt flag */
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_CIFR_HSERDYF                RCC_CIFR_HSERDYF     /*!< HSE Ready Interrupt flag */
#endif
#if defined(RCC_PLL_SUPPORT)
#define LL_RCC_CIFR_PLLRDYF                RCC_CIFR_PLLRDYF     /*!< PLL Ready Interrupt flag */
#endif
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_CIFR_LSERDYF                RCC_CIFR_LSERDYF     /*!< LSE Ready Interrupt flag */
#define LL_RCC_CIFR_LSECSSF                RCC_CIFR_LSECSSF     /*!< LSE Clock Security System Interrupt flag */
#endif
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_CIFR_CSSF                   RCC_CIFR_CSSF        /*!< Clock Security System Interrupt flag */
#endif
#define LL_RCC_CIFR_MSIRDYF                RCC_CIFR_MSIRDYF     /*!< MSI Ready Interrupt flag */
#define LL_RCC_CSR_OBLRSTF                 RCC_CSR_OBLRSTF    /*!< OBL reset flag */
#define LL_RCC_CSR_PINRSTF                 RCC_CSR_PINRSTF    /*!< PIN reset flag */
#define LL_RCC_CSR_SFTRSTF                 RCC_CSR_SFTRSTF    /*!< Software Reset flag */
#define LL_RCC_CSR_IWDGRSTF                RCC_CSR_IWDGRSTF   /*!< Independent Watchdog reset flag */
#if defined(WWDG)
#define LL_RCC_CSR_WWDGRSTF                RCC_CSR_WWDGRSTF   /*!< Window watchdog reset flag */
#endif
#define LL_RCC_CSR_PWRRSTF                 RCC_CSR_PWRRSTF    /*!< BOR or POR/PDR reset flag */
#define LL_RCC_CSR_SPERSTF                 RCC_CSR_SPERSTF    /*!< SRAM1/SRAM2 parity error reset flag */
#define LL_RCC_CSR_LPWRRSTF                RCC_CSR_LPWRRSTF   /*!< Stop and Standby Low Power Mode Reset Flags */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_IT IT Defines
  * @brief    IT defines which can be used with LL_RCC_ReadReg and  LL_RCC_WriteReg functions
  * @{
  */
#define LL_RCC_CIER_LSIRDYIE               RCC_CIER_LSIRDYIE      /*!< LSI Ready Interrupt Enable */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_CIER_LSERDYIE               RCC_CIER_LSERDYIE      /*!< LSE Ready Interrupt Enable */
#endif
#define LL_RCC_CIER_HSIRDYIE               RCC_CIER_HSIRDYIE      /*!< HSI Ready Interrupt Enable */
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_CIER_HSERDYIE               RCC_CIER_HSERDYIE      /*!< HSE Ready Interrupt Enable */
#endif
#if defined(RCC_PLL_SUPPORT)
#define LL_RCC_CIER_PLLRDYIE               RCC_CIER_PLLRDYIE      /*!< PLL Ready Interrupt Enable */
#endif
#define LL_RCC_CIER_MSIRDYIE               RCC_CIER_MSIRDYIE      /*!< MSI Ready Interrupt Enable */
/**
  * @}
  */
  
#if defined(RCC_LSE_SUPPORT)
/** @defgroup RCC_LL_EC_LSEDRIVE  LSE oscillator drive capability
  * @{
  */
#define LL_RCC_LSEDRIVE_LOW                 0x00000000U                              /*!< LSE low drive capability */
#define LL_RCC_LSEDRIVE_MEDIUM              RCC_BDCR_LSEDRV_0                        /*!< LSE medium drive capability */
#define LL_RCC_LSEDRIVE_HIGH                RCC_BDCR_LSEDRV_1                        /*!< LSE high drive capability */
#define LL_RCC_LSEDRIVE_VERY_HIGH           (RCC_BDCR_LSEDRV_1 | RCC_BDCR_LSEDRV_0)  /*!< LSE vrey high drive capability */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_LSESTARTUP  LSE oscillator startup time
  * @{
  */
#define LL_RCC_LSE_STARTUP_NONE         (RCC_BDCR_LSE_RDYSEL_1 | RCC_BDCR_LSE_RDYSEL_0)
#define LL_RCC_LSE_STARTUP_LOW          RCC_BDCR_LSE_RDYSEL_0
#define LL_RCC_LSE_STARTUP_MEDIUM       0x00000000U
#define LL_RCC_LSE_STARTUP_HIGH         RCC_BDCR_LSE_RDYSEL_1
/**
  * @}
  */
#endif

#if defined(RCC_BDCR_LSCSEL)
/** @defgroup RCC_LL_EC_LSC_CLKSOURCE  LSC Selection
  * @{
  */
#define LL_RCC_LSC_CLKSOURCE_LSI           0x00000000U           /*!< LSI selection for low speed clock  */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_LSC_CLKSOURCE_LSE           RCC_BDCR_LSCSEL       /*!< LSE selection for low speed clock  */
#endif
/**
  * @}
  */
#endif

/** @defgroup RCC_LL_EC_SYS_CLKSOURCE  System clock switch
  * @{
  */
#define LL_RCC_SYS_CLKSOURCE_HSISYS        0x00000000U                        /*!< HSISYS selection as system clock */
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_SYS_CLKSOURCE_HSE           RCC_CFGR_SW_0                      /*!< HSE selection as system clock */
#endif
#if defined(RCC_PLL_SUPPORT)
#define LL_RCC_SYS_CLKSOURCE_PLL           RCC_CFGR_SW_1                      /*!< PLL selection as system clock */
#endif
#define LL_RCC_SYS_CLKSOURCE_LSI           (RCC_CFGR_SW_1 | RCC_CFGR_SW_0)    /*!< LSI selection used as system clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_SYS_CLKSOURCE_LSE           RCC_CFGR_SW_2                      /*!< LSE selection used as system clock */
#endif
#define LL_RCC_SYS_CLKSOURCE_MSISYS        (RCC_CFGR_SW_2 | RCC_CFGR_SW_0)    /*!< MSISYS selection used as system clock */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_SYS_CLKSOURCE_STATUS  System clock switch status
  * @{
  */
#define LL_RCC_SYS_CLKSOURCE_STATUS_HSISYS 0x00000000U                         /*!< HSISYS used as system clock */
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_SYS_CLKSOURCE_STATUS_HSE    RCC_CFGR_SWS_0                      /*!< HSE used as system clock */
#endif
#if defined(RCC_PLL_SUPPORT)
#define LL_RCC_SYS_CLKSOURCE_STATUS_PLL    RCC_CFGR_SWS_1                      /*!< PLL used as system clock */
#endif
#define LL_RCC_SYS_CLKSOURCE_STATUS_LSI    (RCC_CFGR_SWS_1 | RCC_CFGR_SWS_0)   /*!< LSI used as system clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_SYS_CLKSOURCE_STATUS_LSE    RCC_CFGR_SWS_2                      /*!< LSE used as system clock */
#endif
#define LL_RCC_SYS_CLKSOURCE_STATUS_MSISYS (RCC_CFGR_SWS_2 | RCC_CFGR_SWS_0)   /*!< MSISYS used as system clock */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_SYSCLK_DIV  AHB prescaler
  * @{
  */
#define LL_RCC_SYSCLK_DIV_1                0x00000000U                                                             /*!< SYSCLK not divided */
#define LL_RCC_SYSCLK_DIV_2                RCC_CFGR_HPRE_3                                                         /*!< SYSCLK divided by 2 */
#define LL_RCC_SYSCLK_DIV_4                (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_0)                                     /*!< SYSCLK divided by 4 */
#define LL_RCC_SYSCLK_DIV_8                (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_1)                                     /*!< SYSCLK divided by 8 */
#define LL_RCC_SYSCLK_DIV_16               (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_1 | RCC_CFGR_HPRE_0)                   /*!< SYSCLK divided by 16 */
#define LL_RCC_SYSCLK_DIV_64               (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_2)                                     /*!< SYSCLK divided by 64 */
#define LL_RCC_SYSCLK_DIV_128              (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_2 | RCC_CFGR_HPRE_0)                   /*!< SYSCLK divided by 128 */
#define LL_RCC_SYSCLK_DIV_256              (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_2 | RCC_CFGR_HPRE_1)                   /*!< SYSCLK divided by 256 */
#define LL_RCC_SYSCLK_DIV_512              (RCC_CFGR_HPRE_3 | RCC_CFGR_HPRE_2 | RCC_CFGR_HPRE_1 | RCC_CFGR_HPRE_0) /*!< SYSCLK divided by 512 */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_APB1_DIV  APB1 low-speed prescaler (APB1)
  * @{
  */
#define LL_RCC_APB1_DIV_1                  0x00000000U                                           /*!< HCLK not divided */
#define LL_RCC_APB1_DIV_2                  RCC_CFGR_PPRE_2                                       /*!< HCLK divided by 2 */
#define LL_RCC_APB1_DIV_4                  (RCC_CFGR_PPRE_2 | RCC_CFGR_PPRE_0)                   /*!< HCLK divided by 4 */
#define LL_RCC_APB1_DIV_8                  (RCC_CFGR_PPRE_2 | RCC_CFGR_PPRE_1)                   /*!< HCLK divided by 8 */
#define LL_RCC_APB1_DIV_16                 (RCC_CFGR_PPRE_2 | RCC_CFGR_PPRE_1 | RCC_CFGR_PPRE_0) /*!< HCLK divided by 16 */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_HSI_DIV  HSI division factor
  * @{
  */
#define LL_RCC_HSI_DIV_1                  0x00000000U                                /*!< HSI not divided */
#define LL_RCC_HSI_DIV_2                  RCC_CR_HSIDIV_0                            /*!< HSI divided by 2 */
#define LL_RCC_HSI_DIV_4                  RCC_CR_HSIDIV_1                            /*!< HSI divided by 4 */
#define LL_RCC_HSI_DIV_8                  (RCC_CR_HSIDIV_1 | RCC_CR_HSIDIV_0)        /*!< HSI divided by 8 */
#define LL_RCC_HSI_DIV_16                 RCC_CR_HSIDIV_2                            /*!< HSI divided by 16 */
#define LL_RCC_HSI_DIV_32                 (RCC_CR_HSIDIV_2 | RCC_CR_HSIDIV_0)        /*!< HSI divided by 32 */
#define LL_RCC_HSI_DIV_64                 (RCC_CR_HSIDIV_2 | RCC_CR_HSIDIV_1)        /*!< HSI divided by 64 */
#define LL_RCC_HSI_DIV_128                RCC_CR_HSIDIV                              /*!< HSI divided by 128 */
/**
  * @}
  */
  
/** @defgroup RCC_LL_EC_MSI_DIV  MSI division factor
  * @{
  */
#define LL_RCC_MSI_DIV_1                  0x00000000U                                        /*!< MSI is not divided */
#define LL_RCC_MSI_DIV_2                  RCC_CR_MSIDIV_0                                    /*!< MSI is divided by 2 */
#define LL_RCC_MSI_DIV_4                  RCC_CR_MSIDIV_1                                    /*!< MSI is divided by 4 */
#define LL_RCC_MSI_DIV_8                  (RCC_CR_MSIDIV_1|RCC_CR_MSIDIV_0)                  /*!< MSI is divided by 8 */
#define LL_RCC_MSI_DIV_16                 RCC_CR_MSIDIV_2                                    /*!< MSI is divided by 16 */
#define LL_RCC_MSI_DIV_32                 (RCC_CR_MSIDIV_2|RCC_CR_MSIDIV_0)                  /*!< MSI is divided by 32 */
#define LL_RCC_MSI_DIV_64                 (RCC_CR_MSIDIV_2|RCC_CR_MSIDIV_1)                  /*!< MSI is divided by 64 */
#define LL_RCC_MSI_DIV_128                (RCC_CR_MSIDIV_2|RCC_CR_MSIDIV_1|RCC_CR_MSIDIV_0)  /*!< MSI is divided by 128 */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_MCOSOURCE  MCO SOURCE selection
  * @{
  */
#define LL_RCC_MCOSOURCE_NOCLOCK          0x00000000U                            /*!< MCO output disabled, no clock on MCO */
#define LL_RCC_MCOSOURCE_SYSCLK           RCC_CFGR_MCOSEL_0                      /*!< SYSCLK selection as MCO source */
#define LL_RCC_MCOSOURCE_HSI10M           RCC_CFGR_MCOSEL_1                      /*!< HSI10M selection as MCO source */
#define LL_RCC_MCOSOURCE_HSI              (RCC_CFGR_MCOSEL_1| RCC_CFGR_MCOSEL_0) /*!< HSI selection as MCO source */
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_MCOSOURCE_HSE              RCC_CFGR_MCOSEL_2                      /*!< HSE selection as MCO source */
#endif
#if defined(RCC_PLL_SUPPORT)
#define LL_RCC_MCOSOURCE_PLLCLK           (RCC_CFGR_MCOSEL_2|RCC_CFGR_MCOSEL_0)  /*!< Main PLL selection as MCO source */
#endif
#define LL_RCC_MCOSOURCE_LSI              (RCC_CFGR_MCOSEL_2|RCC_CFGR_MCOSEL_1)  /*!< LSI selection as MCO source */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_MCOSOURCE_LSE              (RCC_CFGR_MCOSEL_2|RCC_CFGR_MCOSEL_1|RCC_CFGR_MCOSEL_0) /*!< LSE selection as MCO source */
#endif
#define LL_RCC_MCOSOURCE_HCLK             RCC_CFGR_MCOSEL_3                      /*!< HCLK selection as MCO source */
#define LL_RCC_MCOSOURCE_PCLK             (RCC_CFGR_MCOSEL_3| RCC_CFGR_MCOSEL_0) /*!< PCLK selection as MCO source */
#define LL_RCC_MCOSOURCE_MSI              (RCC_CFGR_MCOSEL_3| RCC_CFGR_MCOSEL_1) /*!< MSI selection as MCO source */
#define LL_RCC_MCOSOURCE_LCD_HSI10M       (RCC_CFGR_MCOSEL_3| RCC_CFGR_MCOSEL_1 | RCC_CFGR_MCOSEL_0) /*!< LCD_HSI10M selection as MCO source */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_MCO Peripheral MCO get clock source
  * @{
  */
#define LL_RCC_MCO_CLKSOURCE             RCC_CFGR_MCOSEL /*!< MCO Clock source selection */
/**
  * @}
  */

/** @defgroup RCC_LL_EC_MCO_DIV  MCO prescaler
  * @{
  */
#define LL_RCC_MCO_DIV_1                  (0x00000000U)                                                                   /*!< MCO not divided */
#define LL_RCC_MCO_DIV_2                  (RCC_CFGR_MCOPRE_3)                                                             /*!< MCO divided by 2 */
#define LL_RCC_MCO_DIV_4                  (RCC_CFGR_MCOPRE_3 | RCC_CFGR_MCOPRE_0)                                         /*!< MCO divided by 4 */
#define LL_RCC_MCO_DIV_8                  (RCC_CFGR_MCOPRE_3 | RCC_CFGR_MCOPRE_1)                                         /*!< MCO divided by 8 */
#define LL_RCC_MCO_DIV_16                 (RCC_CFGR_MCOPRE_3 | RCC_CFGR_MCOPRE_1 | RCC_CFGR_MCOPRE_0)                     /*!< MCO divided by 16 */
#define LL_RCC_MCO_DIV_32                 (RCC_CFGR_MCOPRE_3 | RCC_CFGR_MCOPRE_2)                                         /*!< MCO divided by 32 */
#define LL_RCC_MCO_DIV_64                 (RCC_CFGR_MCOPRE_3 | RCC_CFGR_MCOPRE_2 | RCC_CFGR_MCOPRE_0)                     /*!< MCO divided by 64 */
#define LL_RCC_MCO_DIV_128                (RCC_CFGR_MCOPRE_3 | RCC_CFGR_MCOPRE_2 | RCC_CFGR_MCOPRE_1)                     /*!< MCO divided by 128 */
#define LL_RCC_MCO_DIV_256                (RCC_CFGR_MCOPRE_3 | RCC_CFGR_MCOPRE_2 | RCC_CFGR_MCOPRE_1 | RCC_CFGR_MCOPRE_0) /*!< MCO divided by 256 */
/**
  * @}
  */

#if defined(USE_FULL_LL_DRIVER)
/** @defgroup RCC_LL_EC_PERIPH_FREQUENCY Peripheral clock frequency
  * @{
  */
#define LL_RCC_PERIPH_FREQUENCY_NO        0x00000000U                 /*!< No clock enabled for the peripheral            */
#define LL_RCC_PERIPH_FREQUENCY_NA        0xFFFFFFFFU                 /*!< Frequency cannot be provided as external clock */
/**
  * @}
  */
#endif /* USE_FULL_LL_DRIVER */

#if defined(RCC_CCIPR_TIMCLKCTRL)
/** @defgroup RCC_LL_EC_TIM_PCLK_Frequency_Control TIM PCLK Frequency Control
  * @{
  */
#define LL_RCC_TIMPCLK_MUL2               0x00000000U            /*!< TIMER PCLK is twice the system PCLK, but the frequency will not exceed HCLK */
#define LL_RCC_TIMPCLK_MUL1               RCC_CCIPR_TIMCLKCTRL   /*!< TIMER PCLK is the system PCLK */
/**
  * @}
  */
#endif /* RCC_CCIPR_TIM1SEL */

#if defined(RCC_CCIPR_CANSEL)
/** @defgroup RCC_LL_EC_CAN_CLKSOURCE Peripheral CAN clock source selection
  * @{
  */
#define LL_RCC_CAN_CLKSOURCE_PLL        0x00000000U
#define LL_RCC_CAN_CLKSOURCE_PCLK1      RCC_CCIPR_CANSEL_0
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_CAN_CLKSOURCE_HSE        RCC_CCIPR_CANSEL_1
#endif
/**
  * @}
  */
#endif


#if defined(RCC_CCIPR_PVDSEL)
/** @defgroup RCC_LL_EC_PVD_CLKSOURCE Peripheral PVD clock source selection
  * @{
  */
#define LL_RCC_PVD_CLKSOURCE_PCLK1      0x00000000U                   /*!< PCLK1 selected as PVD clock */
#define LL_RCC_PVD_CLKSOURCE_LSC        RCC_CCIPR_PVDSEL              /*!< LSC selected as PVD clock */
/**
  * @}
  */
#endif

#if defined(COMP1)
/** @defgroup RCC_LL_EC_COMPx_CLKSOURCE Peripheral COMP clock source selection
  * @{
  */
#define LL_RCC_COMP1_CLKSOURCE_PCLK1      (RCC_CCIPR_COMP1SEL | (0x00000000U >> 8U))                    /*!< PCLK1 selected as COMP1 clock */
#define LL_RCC_COMP1_CLKSOURCE_LSC        (RCC_CCIPR_COMP1SEL | (RCC_CCIPR_COMP1SEL >> 8U))             /*!< LSC selected as COMP1 clock */
#if defined(COMP2)
#define LL_RCC_COMP2_CLKSOURCE_PCLK1      (RCC_CCIPR_COMP2SEL | (0x00000000U >> 8U))                    /*!< PCLK1 selected as COMP2 clock */
#define LL_RCC_COMP2_CLKSOURCE_LSC        (RCC_CCIPR_COMP2SEL | (RCC_CCIPR_COMP2SEL >> 8U))             /*!< LSC selected as COMP2 clock */
#endif
/**
  * @}
  */
#endif /* COMP1 && COMP2 */

#if defined(RCC_CCIPR_LPUART1SEL)
/** @defgroup RCC_LL_EC_LPUART1x_CLKSOURCE Peripheral LPUART1 clock source selection
  * @{
  */
#define LL_RCC_LPUART1_CLKSOURCE_PCLK1   (RCC_CCIPR_LPUART1SEL | (0x00000000U >> 8U))
#define LL_RCC_LPUART1_CLKSOURCE_SYSCLK  (RCC_CCIPR_LPUART1SEL | (RCC_CCIPR_LPUART1SEL_0 >> 8U))
#define LL_RCC_LPUART1_CLKSOURCE_LSI     (RCC_CCIPR_LPUART1SEL | (RCC_CCIPR_LPUART1SEL_1 >> 8U))
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_LPUART1_CLKSOURCE_LSE     (RCC_CCIPR_LPUART1SEL | (RCC_CCIPR_LPUART1SEL >> 8U))
#endif
/**
  * @}
  */
#endif /* RCC_CCIPR_LPUART1SEL */

#if defined(RCC_CCIPR_LPUART2SEL)
/** @defgroup RCC_LL_EC_LPUART2x_CLKSOURCE Peripheral LPUART2 clock source selection
  * @{
  */
#define LL_RCC_LPUART2_CLKSOURCE_PCLK1   (RCC_CCIPR_LPUART2SEL | (0x00000000U >> 8U))
#define LL_RCC_LPUART2_CLKSOURCE_SYSCLK  (RCC_CCIPR_LPUART2SEL | (RCC_CCIPR_LPUART2SEL_0 >> 8U))
#define LL_RCC_LPUART2_CLKSOURCE_LSI     (RCC_CCIPR_LPUART2SEL | (RCC_CCIPR_LPUART2SEL_1 >> 8U))
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_LPUART2_CLKSOURCE_LSE     (RCC_CCIPR_LPUART2SEL | (RCC_CCIPR_LPUART2SEL >> 8U))
#endif
/**
  * @}
  */
#endif /* RCC_CCIPR_LPUART2SEL */

#if defined(RCC_CCIPR_LPTIM1SEL)
/** @defgroup RCC_LL_EC_LPTIMx_CLKSOURCE Peripheral LPTIM clock source selection
  * @{
  */
#define LL_RCC_LPTIM1_CLKSOURCE_PCLK1      (RCC_CCIPR_LPTIM1SEL | (0x00000000U >> 16U))                   /*!< PCLK1 selected as LPTIM1 clock */
#define LL_RCC_LPTIM1_CLKSOURCE_LSI        (RCC_CCIPR_LPTIM1SEL | (RCC_CCIPR_LPTIM1SEL_0 >> 16U))          /*!< LSI selected as LPTIM1 clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_LPTIM1_CLKSOURCE_LSE        (RCC_CCIPR_LPTIM1SEL | (RCC_CCIPR_LPTIM1SEL >> 16U))            /*!< LSE selected as LPTIM1 clock */
#endif
/**
  * @}
  */
#endif /* RCC_CCIPR_LPTIM1SEL */

#if defined(RCC_CCIPR_LPTIM2SEL)
/** @defgroup RCC_LL_EC_LPTIMx_CLKSOURCE Peripheral LPTIM clock source selection
  * @{
  */
#define LL_RCC_LPTIM2_CLKSOURCE_PCLK1      (RCC_CCIPR_LPTIM2SEL | (0x00000000U >> 16U))                   /*!< PCLK1 selected as LPTIM1 clock */
#define LL_RCC_LPTIM2_CLKSOURCE_LSI        (RCC_CCIPR_LPTIM2SEL | (RCC_CCIPR_LPTIM2SEL_0 >> 16U))          /*!< LSI selected as LPTIM1 clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_LPTIM2_CLKSOURCE_LSE        (RCC_CCIPR_LPTIM2SEL | (RCC_CCIPR_LPTIM2SEL >> 16U))            /*!< LSE selected as LPTIM1 clock */
#endif
/**
  * @}
  */
#endif /* RCC_CCIPR_LPTIM2SEL */

#if defined(RCC_CCIPR_ADCSEL)
/** @defgroup RCC_LL_EC_ADC_CLKSOURCE Peripheral ADC clock source selection
  * @{
  */
#define LL_RCC_ADC_CLKSOURCE_SYSCLK        0x00000000U
#define LL_RCC_ADC_CLKSOURCE_PLL           RCC_CCIPR_ADCSEL_0
#define LL_RCC_ADC_CLKSOURCE_HSI           RCC_CCIPR_ADCSEL_1
/**
  * @}
  */
#endif

#if defined(RCC_BDCR_IWDGSEL)
/** @defgroup RCC_LL_EC_IWDG_CLKSOURCE Peripheral IWDG clock source selection
  * @{
  */
#define LL_RCC_IWDG_CLKSOURCE_LSI        0x00000000U
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_IWDG_CLKSOURCE_LSE        RCC_BDCR_IWDGSEL
#endif
/**
  * @}
  */
#endif

#if defined(RCC_BDCR_RTCSEL)
/** @defgroup RCC_LL_EC_RTC_CLKSOURCE  RTC clock source selection
  * @{
  */
#define LL_RCC_RTC_CLKSOURCE_NONE          0x00000000U             /*!< No clock used as RTC clock */
#if defined(RCC_LSE_SUPPORT)
#define LL_RCC_RTC_CLKSOURCE_LSE           RCC_BDCR_RTCSEL_0       /*!< LSE oscillator clock used as RTC clock */
#endif
#define LL_RCC_RTC_CLKSOURCE_LSI           RCC_BDCR_RTCSEL_1       /*!< LSI oscillator clock used as RTC clock */
#if defined(RCC_HSE_SUPPORT)
#define LL_RCC_RTC_CLKSOURCE_HSE_DIV32    RCC_BDCR_RTCSEL         /*!< HSE oscillator clock divided by 32 used as RTC clock */
#endif
/**
  * @}
  */
#endif

#if defined(RCC_PLL_SUPPORT)

/** @defgroup RCC_LL_EC_PLL_MUL PLL Multiplicator factor
  * @{
  */
#define LL_RCC_PLL_MUL2                    0x00000000U
#define LL_RCC_PLL_MUL3                    RCC_PLLCFGR_PLLMUL_0
#define LL_RCC_PLL_MUL4                    RCC_PLLCFGR_PLLMUL_1
#define LL_RCC_PLL_MUL5                    (RCC_PLLCFGR_PLLMUL_1 | RCC_PLLCFGR_PLLMUL_0)
#define LL_RCC_PLL_MUL6                    RCC_PLLCFGR_PLLMUL_2
#define LL_RCC_PLL_MUL7                    (RCC_PLLCFGR_PLLMUL_2 | RCC_PLLCFGR_PLLMUL_0)
#define LL_RCC_PLL_MUL8                    (RCC_PLLCFGR_PLLMUL_2 | RCC_PLLCFGR_PLLMUL_1)
#define LL_RCC_PLL_MUL9                    (RCC_PLLCFGR_PLLMUL_2 | RCC_PLLCFGR_PLLMUL_1 | RCC_PLLCFGR_PLLMUL_0)
#define LL_RCC_PLL_MUL10                   RCC_PLLCFGR_PLLMUL_3
#define LL_RCC_PLL_MUL11                   (RCC_PLLCFGR_PLLMUL_3 | RCC_PLLCFGR_PLLMUL_0)
#define LL_RCC_PLL_MUL12                   (RCC_PLLCFGR_PLLMUL_3 | RCC_PLLCFGR_PLLMUL_1)
#define LL_RCC_PLL_MUL13                   (RCC_PLLCFGR_PLLMUL_3 | RCC_PLLCFGR_PLLMUL_1 | RCC_PLLCFGR_PLLMUL_0)
#define LL_RCC_PLL_MUL14                   (RCC_PLLCFGR_PLLMUL_3 | RCC_PLLCFGR_PLLMUL_2)
#define LL_RCC_PLL_MUL15                   (RCC_PLLCFGR_PLLMUL_3 | RCC_PLLCFGR_PLLMUL_2 | RCC_PLLCFGR_PLLMUL_0)
#define LL_RCC_PLL_MUL16                   (RCC_PLLCFGR_PLLMUL_3 | RCC_PLLCFGR_PLLMUL_2 | RCC_PLLCFGR_PLLMUL_1)
#define LL_RCC_PLL_MUL17                   (RCC_PLLCFGR_PLLMUL_3 | RCC_PLLCFGR_PLLMUL_2 | RCC_PLLCFGR_PLLMUL_1 | RCC_PLLCFGR_PLLMUL_0)
#define LL_RCC_PLL_MUL18                   RCC_PLLCFGR_PLLMUL_4
/**
  * @}
  */

/** @defgroup RCC_LL_EC_PLLSOURCE  PLL entry clock source
  * @{
  */
#define LL_RCC_PLLSOURCE_HSI               0x00000000U         /*!< HSI clock selected as PLL entry clock source */
#if defined(RCC_HSE_SUPPORT)               
#define LL_RCC_PLLSOURCE_HSE               RCC_PLLCFGR_PLLSRC  /*!< HSE clock selected as PLL entry clock source */
#endif
/**
  * @}
  */
#endif

#if defined(COMP1)
/** @defgroup RCC_LL_EC_COMP Peripheral COMP get clock source
  * @{
  */
#define LL_RCC_COMP1_CLKSOURCE             RCC_CCIPR_COMP1SEL /*!< COMP1 Clock source selection */
#if defined(COMP2)
#define LL_RCC_COMP2_CLKSOURCE             RCC_CCIPR_COMP2SEL /*!< COMP2 Clock source selection */
#endif /* COMP2 */
/**
  * @}
  */
#endif /* COMP1 */

#if defined(RCC_CCIPR_LPUART1SEL)
/** @defgroup RCC_LL_EC_LPUART1 Peripheral LPUART1 get clock source
  * @{
  */
#define LL_RCC_LPUART1_CLKSOURCE            RCC_CCIPR_LPUART1SEL /*!< LPUART1 Clock source selection */
/**
  * @}
  */
#endif /* RCC_CCIPR_LPUART1SEL */

#if defined(RCC_CCIPR_LPUART2SEL)
/** @defgroup RCC_LL_EC_LPUART2 Peripheral LPUART2 get clock source
  * @{
  */
#define LL_RCC_LPUART2_CLKSOURCE            RCC_CCIPR_LPUART2SEL /*!< LPUART2 Clock source selection */
/**
  * @}
  */
#endif /* RCC_CCIPR_LPUART2SEL */

#if defined(RCC_CCIPR_LPTIM1SEL)
/** @defgroup RCC_LL_EC_LPTIM Peripheral LPTIM get clock source
  * @{
  */
#define LL_RCC_LPTIM1_CLKSOURCE            RCC_CCIPR_LPTIM1SEL /*!< LPTIM1 Clock source selection */
/**
  * @}
  */
#endif /* RCC_CCIPR_LPTIM1SEL */

#if defined(RCC_CCIPR_LPTIM2SEL)
/** @defgroup RCC_LL_EC_LPTIM Peripheral LPTIM get clock source
  * @{
  */
#define LL_RCC_LPTIM2_CLKSOURCE            RCC_CCIPR_LPTIM2SEL /*!< LPTIM1 Clock source selection */
/**
  * @}
  */
#endif /* RCC_CCIPR_LPTIM2SEL */

#if defined(RCC_HSE_SUPPORT)
/** @defgroup RCC_HSE_EC_Freq HSE Config
  * @{
  */
#define LL_RCC_HSE_4_8MHz                 0x00000000U                                  /*!< HSE low drive capability */
#define LL_RCC_HSE_8_16MHz                RCC_ECSCR_HSE_DRV_0                          /*!< HSE medium drive capability */
#define LL_RCC_HSE_16_24MHz               RCC_ECSCR_HSE_DRV_1                          /*!< HSE hign drive capability */
#define LL_RCC_HSE_24_32MHz               RCC_ECSCR_HSE_DRV                            /*!< HSE very high drive capability */
/**
  * @}
  */

/** @defgroup RCC_HSE_EC_STARTUP HSE settling time Config
  * @{
  */
#define LL_RCC_HSE_STARTUP_NONE         (RCC_ECSCR_HSE_RDYSEL_1 | RCC_ECSCR_HSE_RDYSEL_0)
#define LL_RCC_HSE_STARTUP_LOW          RCC_ECSCR_HSE_RDYSEL_0
#define LL_RCC_HSE_STARTUP_MEDIUM       0x00000000U
#define LL_RCC_HSE_STARTUP_HIGH         RCC_ECSCR_HSE_RDYSEL_1
/**
  * @}
  */
#endif
/** @defgroup RCC_HSI_EC_Calibration HSI Calibration
* @{
*/
#define LL_RCC_HSICALIBRATION_8MHz        ((0x1<<13) | ((*(uint32_t *)(0x1FFF1E04)) & 0x1FFF))  /*!< 8MHz HSI calibration trimming value */
#define LL_RCC_HSICALIBRATION_16MHz       ((0x2<<13) | ((*(uint32_t *)(0x1FFF1E08)) & 0x1FFF))  /*!< 16MHz HSI calibration trimming value */
#define LL_RCC_HSICALIBRATION_22p12MHz    ((0x3<<13) | ((*(uint32_t *)(0x1FFF1E0C)) & 0x1FFF))  /*!< 22.12MHz HSI calibration trimming value */
#define LL_RCC_HSICALIBRATION_24MHz       ((0x4<<13) | ((*(uint32_t *)(0x1FFF1E10)) & 0x1FFF))  /*!< 24MHz HSI calibration trimming value */
#define LL_RCC_HSICALIBRATION_48MHz       ((0x5<<13) | ((*(uint32_t *)(0x1FFF1E14)) & 0x1FFF))  /*!< 48MHz HSI calibration trimming value */
#define LL_RCC_HSICALIBRATION_64MHz       ((0x6<<13) | ((*(uint32_t *)(0x1FFF1E18)) & 0x1FFF))  /*!< 64MHz HSI calibration trimming value */
/**
  * @}
  */

/**
  * @}
  */

/* Exported macro ------------------------------------------------------------*/
/** @defgroup RCC_LL_Exported_Macros RCC Exported Macros
  * @{
  */

/** @defgroup RCC_LL_EM_WRITE_READ Common Write and read registers Macros
  * @{
  */

/**
  * @brief  Write a value in RCC register
  * @param  __REG__ Register to be written
  * @param  __VALUE__ Value to be written in the register
  * @retval None
  */
#define LL_RCC_WriteReg(__REG__, __VALUE__) WRITE_REG((RCC->__REG__), (__VALUE__))

/**
  * @brief  Read a value in RCC register
  * @param  __REG__ Register to be read
  * @retval Register value
  */
#define LL_RCC_ReadReg(__REG__) READ_REG(RCC->__REG__)
/**
  * @}
  */

/** @defgroup RCC_LL_EM_CALC_FREQ Calculate frequencies
  * @{
  */
#if defined(RCC_PLL_SUPPORT)
/**
  * @brief  Helper macro to calculate the PLLCLK frequency
  * @param  __INPUTFREQ__ PLL Input frequency (based on HSE / HSI)
  * @param  __PLLMUL__ This parameter can be one of the following values:
  *         @arg @ref LL_RCC_PLL_MUL2 
  *         @arg @ref LL_RCC_PLL_MUL3 
  *         @arg @ref LL_RCC_PLL_MUL4 
  *         @arg @ref LL_RCC_PLL_MUL5 
  *         @arg @ref LL_RCC_PLL_MUL6 
  *         @arg @ref LL_RCC_PLL_MUL7 
  *         @arg @ref LL_RCC_PLL_MUL8 
  *         @arg @ref LL_RCC_PLL_MUL9 
  *         @arg @ref LL_RCC_PLL_MUL10
  *         @arg @ref LL_RCC_PLL_MUL11
  *         @arg @ref LL_RCC_PLL_MUL12
  *         @arg @ref LL_RCC_PLL_MUL13
  *         @arg @ref LL_RCC_PLL_MUL14
  *         @arg @ref LL_RCC_PLL_MUL15
  *         @arg @ref LL_RCC_PLL_MUL16
  *         @arg @ref LL_RCC_PLL_MUL17
  *         @arg @ref LL_RCC_PLL_MUL18
  * @retval PLL clock frequency (in Hz)
  */
#define __LL_RCC_CALC_PLLCLK_FREQ(__INPUTFREQ__, __PLLMUL__) \
          ((__INPUTFREQ__) * ((((__PLLMUL__) & RCC_PLLCFGR_PLLMUL) >> RCC_PLLCFGR_PLLMUL_Pos) + 2U))
#endif

/**
  * @brief  Helper macro to calculate the HCLK frequency
  * @param  __SYSCLKFREQ__ SYSCLK frequency
  * @param  __AHBPRESCALER__ This parameter can be one of the following values:
  *         @arg @ref LL_RCC_SYSCLK_DIV_1
  *         @arg @ref LL_RCC_SYSCLK_DIV_2
  *         @arg @ref LL_RCC_SYSCLK_DIV_4
  *         @arg @ref LL_RCC_SYSCLK_DIV_8
  *         @arg @ref LL_RCC_SYSCLK_DIV_16
  *         @arg @ref LL_RCC_SYSCLK_DIV_64
  *         @arg @ref LL_RCC_SYSCLK_DIV_128
  *         @arg @ref LL_RCC_SYSCLK_DIV_256
  *         @arg @ref LL_RCC_SYSCLK_DIV_512
  * @retval HCLK clock frequency (in Hz)
  */
#define __LL_RCC_CALC_HCLK_FREQ(__SYSCLKFREQ__,__AHBPRESCALER__)  \
  ((__SYSCLKFREQ__) >> (AHBPrescTable[((__AHBPRESCALER__) & RCC_CFGR_HPRE) >>  RCC_CFGR_HPRE_Pos] & 0x1FU))

/**
  * @brief  Helper macro to calculate the PCLK1 frequency (APB1)
  * @param  __HCLKFREQ__ HCLK frequency
  * @param  __APB1PRESCALER__ This parameter can be one of the following values:
  *         @arg @ref LL_RCC_APB1_DIV_1
  *         @arg @ref LL_RCC_APB1_DIV_2
  *         @arg @ref LL_RCC_APB1_DIV_4
  *         @arg @ref LL_RCC_APB1_DIV_8
  *         @arg @ref LL_RCC_APB1_DIV_16
  * @retval PCLK1 clock frequency (in Hz)
  */
#define __LL_RCC_CALC_PCLK1_FREQ(__HCLKFREQ__, __APB1PRESCALER__)  \
  ((__HCLKFREQ__) >> (APBPrescTable[(__APB1PRESCALER__) >>  RCC_CFGR_PPRE_Pos] & 0x1FU))

/**
  * @brief  Helper macro to calculate the HSISYS frequency
  * @param  __HSIDIV__ This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSI_DIV_1
  *         @arg @ref LL_RCC_HSI_DIV_2
  *         @arg @ref LL_RCC_HSI_DIV_4
  *         @arg @ref LL_RCC_HSI_DIV_8
  *         @arg @ref LL_RCC_HSI_DIV_16
  *         @arg @ref LL_RCC_HSI_DIV_32
  *         @arg @ref LL_RCC_HSI_DIV_64
  *         @arg @ref LL_RCC_HSI_DIV_128
  * @retval HSISYS clock frequency (in Hz)
  */
#define __LL_RCC_CALC_HSI_FREQ(__HSIDIV__)  \
  (HSIFreqTable[(RCC->ICSCR & RCC_ICSCR_HSI_FS_OPCR) >> RCC_ICSCR_HSI_FS_OPCR_Pos] / (1U << ((__HSIDIV__)>> RCC_CR_HSIDIV_Pos)))
  
/**
  * @brief  Helper macro to calculate the MSISYS frequency
  * @param  __MSIDIV__ This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MSI_DIV_1
  *         @arg @ref LL_RCC_MSI_DIV_2
  *         @arg @ref LL_RCC_MSI_DIV_4
  *         @arg @ref LL_RCC_MSI_DIV_8
  *         @arg @ref LL_RCC_MSI_DIV_16
  *         @arg @ref LL_RCC_MSI_DIV_32
  *         @arg @ref LL_RCC_MSI_DIV_64
  *         @arg @ref LL_RCC_MSI_DIV_128
  * @retval MSISYS clock frequency (in Hz)
  */
#define __LL_RCC_CALC_MSI_FREQ(__MSIDIV__)  \
  (MSI_VALUE / (1U << ((__MSIDIV__)>> RCC_CR_MSIDIV_Pos)))

/**
  * @}
  */

/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/
/** @defgroup RCC_LL_Exported_Functions RCC Exported Functions
  * @{
  */

#if defined(RCC_HSE_SUPPORT)
/** @defgroup RCC_LL_EF_HSE HSE
  * @{
  */

/**
  * @brief  Enable the Clock Security System.
  * @rmtoll CR           HSE_CSSON         LL_RCC_HSE_EnableCSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_EnableCSS(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSE_CSSON);
}

/**
  * @brief  Enable HSE external oscillator (HSE Bypass)
  * @rmtoll CR           HSEBYP        LL_RCC_HSE_EnableBypass
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_EnableBypass(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSEBYP);
}

/**
  * @brief  Disable HSE external oscillator (HSE Bypass)
  * @rmtoll CR           HSEBYP        LL_RCC_HSE_DisableBypass
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_DisableBypass(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_HSEBYP);
}

/**
  * @brief  Enable HSE crystal oscillator (HSE ON)
  * @rmtoll CR           HSEON         LL_RCC_HSE_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_Enable(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSEON);
}

/**
  * @brief  Disable HSE crystal oscillator (HSE ON)
  * @rmtoll CR           HSEON         LL_RCC_HSE_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_Disable(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_HSEON);
}

/**
  * @brief  Check if HSE oscillator Ready
  * @rmtoll CR           HSERDY        LL_RCC_HSE_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_HSE_IsReady(void)
{
  return ((READ_BIT(RCC->CR, RCC_CR_HSERDY) == (RCC_CR_HSERDY)) ? 1UL : 0UL);
}

/**
  * @brief  Set HSE Crystal working frequency
  * @param  HSEFreq This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSE_4_8MHz
  *         @arg @ref LL_RCC_HSE_8_16MHz
  *         @arg @ref LL_RCC_HSE_16_24MHz
  *         @arg @ref LL_RCC_HSE_24_32MHz
  * @note   The maximum working frequency varies depending on the devices.
  *         Refer to the devices data sheet for the maximum working frequency.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_SetFreqRegion(uint32_t HSEFreq)
{
  MODIFY_REG(RCC->ECSCR, RCC_ECSCR_HSE_DRV, HSEFreq);
}

/**
  * @brief  Set HSE settling time.
  * @param  Time This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSE_STARTUP_NONE
  *         @arg @ref LL_RCC_HSE_STARTUP_LOW
  *         @arg @ref LL_RCC_HSE_STARTUP_MEDIUM
  *         @arg @ref LL_RCC_HSE_STARTUP_HIGH
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_SetStartupTime(uint32_t Time)
{
  MODIFY_REG(RCC->ECSCR, RCC_ECSCR_HSE_RDYSEL, Time);
}

/**
  * @brief  Get HSE settling time.
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_HSE_STARTUP_NONE
  *         @arg @ref LL_RCC_HSE_STARTUP_LOW
  *         @arg @ref LL_RCC_HSE_STARTUP_MEDIUM
  *         @arg @ref LL_RCC_HSE_STARTUP_HIGH
  */
__STATIC_INLINE uint32_t LL_RCC_HSE_GetStartupTime(void)
{
  return (uint32_t)(READ_BIT(RCC->ECSCR, RCC_ECSCR_HSE_RDYSEL));
}

/**
  * @brief  Enable filtering of HSE.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_EnableFilter(void)
{
  CLEAR_BIT(RCC->ECSCR, RCC_ECSCR_HSE_FILT_ENB);
}

/**
  * @brief  Disable filtering of HSE.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSE_DisableFilter(void)
{
  SET_BIT(RCC->ECSCR, RCC_ECSCR_HSE_FILT_ENB);
}


/**
  * @}
  */
#endif

/** @defgroup RCC_LL_EF_HSI HSI
  * @{
  */

/**
  * @brief  Enable HSI even in stop mode
  * @note   HSI oscillator is forced ON even in Stop mode
  * @rmtoll CR           HSIKERON      LL_RCC_HSI_EnableInStopMode
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_EnableInStopMode(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSIKERON);
}

/**
  * @brief  Disable HSI in stop mode
  * @rmtoll CR           HSIKERON      LL_RCC_HSI_DisableInStopMode
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_DisableInStopMode(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_HSIKERON);
}

/**
  * @brief  Check if HSI is enabled in stop mode
  * @rmtoll CR           HSIKERON        LL_RCC_HSI_IsEnabledInStopMode
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_HSI_IsEnabledInStopMode(void)
{
  return ((READ_BIT(RCC->CR, RCC_CR_HSIKERON) == RCC_CR_HSIKERON) ? 1UL : 0UL);
}

/**
  * @brief  Enable HSI oscillator
  * @rmtoll CR           HSION         LL_RCC_HSI_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_Enable(void)
{
  SET_BIT(RCC->CR, RCC_CR_HSION);
}

/**
  * @brief  Disable HSI oscillator
  * @rmtoll CR           HSION         LL_RCC_HSI_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_Disable(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_HSION);
}

/**
  * @brief  Check if HSI clock is ready
  * @rmtoll CR           HSIRDY        LL_RCC_HSI_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_HSI_IsReady(void)
{
  return ((READ_BIT(RCC->CR, RCC_CR_HSIRDY) == (RCC_CR_HSIRDY)) ? 1UL : 0UL);
}

/**
  * @brief  Set HSI Calibration trimming
  * @param  Value Between Min_Data = 0 and Max_Data = 0x1FFF
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_SetCalibTrimming(uint32_t Value)
{
  MODIFY_REG(RCC->ICSCR, RCC_ICSCR_HSI_TRIMCR, Value << RCC_ICSCR_HSI_TRIMCR_Pos);
}

/**
  * @brief  Get HSI Calibration trimming
  * @rmtoll ICSCR        HSITRIM       LL_RCC_HSI_GetCalibTrimming
  * @retval Between Min_Data = 0 and Max_Data = 0x1FFF
  */
__STATIC_INLINE uint32_t LL_RCC_HSI_GetCalibTrimming(void)
{
  return (uint32_t)(READ_BIT(RCC->ICSCR, RCC_ICSCR_HSI_TRIMCR) >> RCC_ICSCR_HSI_TRIMCR_Pos);
}

/**
  * @brief  Set HSI Calibration Frequency
  * @param  Value This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSICALIBRATION_8MHz    
  *         @arg @ref LL_RCC_HSICALIBRATION_16MHz   
  *         @arg @ref LL_RCC_HSICALIBRATION_22p12MHz
  *         @arg @ref LL_RCC_HSICALIBRATION_24MHz   
  *         @arg @ref LL_RCC_HSICALIBRATION_48MHz   
  *         @arg @ref LL_RCC_HSICALIBRATION_64MHz   
  * @note   Depending on devices and packages, some calibration values may not be available.
  *         Refer to device datasheet for calibration values availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_HSI_SetCalibFreq(uint32_t Value)
{
  MODIFY_REG(RCC->ICSCR, (RCC_ICSCR_HSI_FS_OPCR | RCC_ICSCR_HSI_TRIMCR), Value);
}

/**
  * @brief  Get HSI Frequency
  * @retval HSI clock frequency (in Hz)
  */
__STATIC_INLINE uint32_t LL_RCC_HSI_GetFreq(void)
{
  return (uint32_t)HSIFreqTable[(RCC->ICSCR & RCC_ICSCR_HSI_FS_OPCR) >> RCC_ICSCR_HSI_FS_OPCR_Pos];
}

/**
  * @}
  */

/** @defgroup RCC_LL_EF_MSI MSI
  * @{
  */
/**
  * @brief  Enable MSI oscillator
  * @retval None
  */
__STATIC_INLINE void LL_RCC_MSI_Enable(void)
{
  SET_BIT(RCC->CR, RCC_CR_MSION);
}

/**
  * @brief  Disable MSI oscillator
  * @retval None
  */
__STATIC_INLINE void LL_RCC_MSI_Disable(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_MSION);
}

/**
  * @brief  Check if MSI clock is ready
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_MSI_IsReady(void)
{
  return ((READ_BIT(RCC->CR, RCC_CR_MSIRDY) == (RCC_CR_MSIRDY)) ? 1UL : 0UL);
}
/**
  * @}
  */

#if defined(RCC_LSE_SUPPORT)
/** @defgroup RCC_LL_EF_LSE LSE
  * @{
  */

/**
  * @brief  Enable  Low Speed External (LSE) crystal.
  * @rmtoll BDCR         LSEON         LL_RCC_LSE_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_Enable(void)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_LSEEN, (RCC_BDCR_LSEEN_2 | RCC_BDCR_LSEEN_0));
}

/**
  * @brief  Disable  Low Speed External (LSE) crystal.
  * @rmtoll BDCR         LSEON         LL_RCC_LSE_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_Disable(void)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_LSEEN, (RCC_BDCR_LSEEN_3 | RCC_BDCR_LSEEN_1));
}

/**
  * @brief  Enable external clock source (LSE bypass).
  * @rmtoll BDCR         LSEBYP        LL_RCC_LSE_EnableBypass
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_EnableBypass(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_LSEBYP);
}

/**
  * @brief  Disable external clock source (LSE bypass).
  * @rmtoll BDCR         LSEBYP        LL_RCC_LSE_DisableBypass
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_DisableBypass(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_LSEBYP);
}

/**
  * @brief  Set LSE oscillator drive capability
  * @note The oscillator is in Xtal mode when it is not in bypass mode.
  * @param  LSEDrive This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LSEDRIVE_LOW
  *         @arg @ref LL_RCC_LSEDRIVE_MEDIUM
  *         @arg @ref LL_RCC_LSEDRIVE_HIGH
  *         @arg @ref LL_RCC_LSEDRIVE_VERY_HIGH
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_SetDriveCapability(uint32_t LSEDrive)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_LSEDRV, LSEDrive);
}

/**
  * @brief  Get LSE oscillator drive capability
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_LSEDRIVE_LOW
  *         @arg @ref LL_RCC_LSEDRIVE_MEDIUM
  *         @arg @ref LL_RCC_LSEDRIVE_HIGH
  *         @arg @ref LL_RCC_LSEDRIVE_VERY_HIGH
  */
__STATIC_INLINE uint32_t LL_RCC_LSE_GetDriveCapability(void)
{
  return (uint32_t)(READ_BIT(RCC->BDCR, RCC_BDCR_LSEDRV));
}

/**
  * @brief  Set LSE startup time
  * @param  LSEStartup This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LSE_STARTUP_NONE     
  *         @arg @ref LL_RCC_LSE_STARTUP_LOW      
  *         @arg @ref LL_RCC_LSE_STARTUP_MEDIUM   
  *         @arg @ref LL_RCC_LSE_STARTUP_HIGH     
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_SetStartupTime(uint32_t LSEStartup)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_LSE_RDYSEL, LSEStartup);
}

/**
  * @brief  Get LSE startup time
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_LSE_STARTUP_NONE     
  *         @arg @ref LL_RCC_LSE_STARTUP_LOW      
  *         @arg @ref LL_RCC_LSE_STARTUP_MEDIUM   
  *         @arg @ref LL_RCC_LSE_STARTUP_HIGH     
  */
__STATIC_INLINE uint32_t LL_RCC_LSE_GetStartupTime(void)
{
  return (uint32_t)(READ_BIT(RCC->BDCR, RCC_BDCR_LSE_RDYSEL));
}

/**
  * @brief  Enable Clock security system on LSE.
  * @rmtoll BDCR         LSECSSON      LL_RCC_LSE_EnableCSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_EnableCSS(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_LSECSSON);
}

/**
  * @brief  Disable Clock security system on LSE.
  * @note Clock security system can be disabled only after a LSE
  *       failure detection. In that case it MUST be disabled by software.
  * @rmtoll BDCR         LSECSSON      LL_RCC_LSE_DisableCSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSE_DisableCSS(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_LSECSSON);
}


/**
  * @brief  Check if LSE oscillator Ready
  * @rmtoll BDCR         LSERDY        LL_RCC_LSE_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_LSE_IsReady(void)
{
  return ((READ_BIT(RCC->BDCR, RCC_BDCR_LSERDY) == (RCC_BDCR_LSERDY)) ? 1UL : 0UL);
}

/**
  * @brief  Check if CSS on LSE failure Detection
  * @rmtoll BDCR         LSECSSD       LL_RCC_LSE_IsCSSDetected
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_LSE_IsCSSDetected(void)
{
  return ((READ_BIT(RCC->BDCR, RCC_BDCR_LSECSSD) == (RCC_BDCR_LSECSSD)) ? 1UL : 0UL);
}

/**
  * @}
  */
#endif

/** @defgroup RCC_LL_EF_LSI LSI
  * @{
  */

/**
  * @brief  Enable LSI Oscillator
  * @rmtoll CSR          LSION         LL_RCC_LSI_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSI_Enable(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_LSION);
}

/**
  * @brief  Disable LSI Oscillator
  * @rmtoll CSR          LSION         LL_RCC_LSI_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSI_Disable(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_LSION);
}

/**
  * @brief  Check if LSI is Ready
  * @rmtoll CSR          LSIRDY        LL_RCC_LSI_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_LSI_IsReady(void)
{
  return ((READ_BIT(RCC->BDCR, RCC_BDCR_LSIRDY) == (RCC_BDCR_LSIRDY)) ? 1UL : 0UL);
}

/**
  * @brief  Set LSI Calibration trimming
  * @param  Value Between Min_Data = 0 and Max_Data = 0x1FF
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSI_SetCalibTrimming(uint32_t Value)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_LSI_TRIMCR, Value << RCC_BDCR_LSI_TRIMCR_Pos);
}

/**
  * @brief  Get LSI Calibration trimming
  * @retval Between Min_Data = 0 and Max_Data = 0x1FF
  */
__STATIC_INLINE uint32_t LL_RCC_LSI_GetCalibTrimming(void)
{
  return (uint32_t)(READ_BIT(RCC->BDCR, RCC_BDCR_LSI_TRIMCR) >> RCC_BDCR_LSI_TRIMCR_Pos);
}

/**
  * @}
  */
#if defined(RCC_BDCR_LSCSEL)
/** @defgroup RCC_LL_EF_LSC LSC
  * @{
  */
/**
  * @brief  Configure Low speed clock selection
  * @rmtoll BDCR         LSCSEL       LL_RCC_LSC_SetSource
  * @param  Source This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LSC_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LSC_CLKSOURCE_LSE
  * @retval None
  */
__STATIC_INLINE void LL_RCC_LSC_SetSource(uint32_t Source)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_LSCSEL, Source);
}

/**
  * @brief  Get Low speed clock selection
  * @rmtoll BDCR         LSCSEL       LL_RCC_LSC_GetSource
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_LSC_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LSC_CLKSOURCE_LSE
  */
__STATIC_INLINE uint32_t LL_RCC_LSC_GetSource(void)
{
  return (uint32_t)(READ_BIT(RCC->BDCR, RCC_BDCR_LSCSEL));
}
/**
  * @}
  */
#endif

/** @defgroup RCC_LL_EF_System System
  * @{
  */

/**
  * @brief  Configure the system clock source
  * @rmtoll CFGR         SW            LL_RCC_SetSysClkSource
  * @param  Source This parameter can be one of the following values:
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_HSISYS
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_HSE
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_PLL
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_MSISYS
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetSysClkSource(uint32_t Source)
{
  MODIFY_REG(RCC->CFGR, RCC_CFGR_SW, Source);
}

/**
  * @brief  Get the system clock source
  * @rmtoll CFGR         SWS           LL_RCC_GetSysClkSource
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_HSISYS
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_HSE
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_PLL
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_LSI
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_LSE
  *         @arg @ref LL_RCC_SYS_CLKSOURCE_STATUS_MSISYS
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetSysClkSource(void)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, RCC_CFGR_SWS));
}

/**
  * @brief  Set AHB prescaler
  * @rmtoll CFGR         HPRE          LL_RCC_SetAHBPrescaler
  * @param  Prescaler This parameter can be one of the following values:
  *         @arg @ref LL_RCC_SYSCLK_DIV_1
  *         @arg @ref LL_RCC_SYSCLK_DIV_2
  *         @arg @ref LL_RCC_SYSCLK_DIV_4
  *         @arg @ref LL_RCC_SYSCLK_DIV_8
  *         @arg @ref LL_RCC_SYSCLK_DIV_16
  *         @arg @ref LL_RCC_SYSCLK_DIV_64
  *         @arg @ref LL_RCC_SYSCLK_DIV_128
  *         @arg @ref LL_RCC_SYSCLK_DIV_256
  *         @arg @ref LL_RCC_SYSCLK_DIV_512
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetAHBPrescaler(uint32_t Prescaler)
{
  MODIFY_REG(RCC->CFGR, RCC_CFGR_HPRE, Prescaler);
}

/**
  * @brief  Set APB1 prescaler
  * @rmtoll CFGR         PPRE         LL_RCC_SetAPB1Prescaler
  * @param  Prescaler This parameter can be one of the following values:
  *         @arg @ref LL_RCC_APB1_DIV_1
  *         @arg @ref LL_RCC_APB1_DIV_2
  *         @arg @ref LL_RCC_APB1_DIV_4
  *         @arg @ref LL_RCC_APB1_DIV_8
  *         @arg @ref LL_RCC_APB1_DIV_16
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetAPB1Prescaler(uint32_t Prescaler)
{
  MODIFY_REG(RCC->CFGR, RCC_CFGR_PPRE, Prescaler);
}

/**
  * @brief  Set HSI division factor
  * @rmtoll CR         HSIDIV          LL_RCC_SetHSIDiv
  * @note  HSIDIV parameter is only applied to SYSCLK_Frequency when HSI is used as
  * system clock source.
  * @param  HSIDiv  This parameter can be one of the following values:
  *         @arg @ref LL_RCC_HSI_DIV_1
  *         @arg @ref LL_RCC_HSI_DIV_2
  *         @arg @ref LL_RCC_HSI_DIV_4
  *         @arg @ref LL_RCC_HSI_DIV_8
  *         @arg @ref LL_RCC_HSI_DIV_16
  *         @arg @ref LL_RCC_HSI_DIV_32
  *         @arg @ref LL_RCC_HSI_DIV_64
  *         @arg @ref LL_RCC_HSI_DIV_128
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetHSIDiv(uint32_t HSIDiv)
{
  MODIFY_REG(RCC->CR, RCC_CR_HSIDIV, HSIDiv);
}

/**
  * @brief  Set MSI division factor
  * @param  MSIDiv  This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MSI_DIV_1
  *         @arg @ref LL_RCC_MSI_DIV_2
  *         @arg @ref LL_RCC_MSI_DIV_4
  *         @arg @ref LL_RCC_MSI_DIV_8
  *         @arg @ref LL_RCC_MSI_DIV_16
  *         @arg @ref LL_RCC_MSI_DIV_32
  *         @arg @ref LL_RCC_MSI_DIV_64
  *         @arg @ref LL_RCC_MSI_DIV_128
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetMSIDiv(uint32_t MSIDiv)
{
  MODIFY_REG(RCC->CR, RCC_CR_MSIDIV, MSIDiv);
}

/**
  * @brief  Get AHB prescaler
  * @rmtoll CFGR         HPRE          LL_RCC_GetAHBPrescaler
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_SYSCLK_DIV_1
  *         @arg @ref LL_RCC_SYSCLK_DIV_2
  *         @arg @ref LL_RCC_SYSCLK_DIV_4
  *         @arg @ref LL_RCC_SYSCLK_DIV_8
  *         @arg @ref LL_RCC_SYSCLK_DIV_16
  *         @arg @ref LL_RCC_SYSCLK_DIV_64
  *         @arg @ref LL_RCC_SYSCLK_DIV_128
  *         @arg @ref LL_RCC_SYSCLK_DIV_256
  *         @arg @ref LL_RCC_SYSCLK_DIV_512
  */
__STATIC_INLINE uint32_t LL_RCC_GetAHBPrescaler(void)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, RCC_CFGR_HPRE));
}

/**
  * @brief  Get APB1 prescaler
  * @rmtoll CFGR         PPRE         LL_RCC_GetAPB1Prescaler
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_APB1_DIV_1
  *         @arg @ref LL_RCC_APB1_DIV_2
  *         @arg @ref LL_RCC_APB1_DIV_4
  *         @arg @ref LL_RCC_APB1_DIV_8
  *         @arg @ref LL_RCC_APB1_DIV_16
  */
__STATIC_INLINE uint32_t LL_RCC_GetAPB1Prescaler(void)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, RCC_CFGR_PPRE));
}

/**
  * @brief  Get HSI division factor
  * @rmtoll CR         HSIDIV         LL_RCC_GetHSIDiv
  * @note  HSIDIV parameter is only applied to SYSCLK_Frequency when HSI is used as
  * system clock source.
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_HSI_DIV_1
  *         @arg @ref LL_RCC_HSI_DIV_2
  *         @arg @ref LL_RCC_HSI_DIV_4
  *         @arg @ref LL_RCC_HSI_DIV_8
  *         @arg @ref LL_RCC_HSI_DIV_16
  *         @arg @ref LL_RCC_HSI_DIV_32
  *         @arg @ref LL_RCC_HSI_DIV_64
  *         @arg @ref LL_RCC_HSI_DIV_128
  */
__STATIC_INLINE uint32_t LL_RCC_GetHSIDiv(void)
{
  return (uint32_t)(READ_BIT(RCC->CR, RCC_CR_HSIDIV));
}

/**
  * @brief  Get MSI division factor
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_MSI_DIV_1
  *         @arg @ref LL_RCC_MSI_DIV_2
  *         @arg @ref LL_RCC_MSI_DIV_4
  *         @arg @ref LL_RCC_MSI_DIV_8
  *         @arg @ref LL_RCC_MSI_DIV_16
  *         @arg @ref LL_RCC_MSI_DIV_32
  *         @arg @ref LL_RCC_MSI_DIV_64
  *         @arg @ref LL_RCC_MSI_DIV_128
  */
__STATIC_INLINE uint32_t LL_RCC_GetMSIDiv(void)
{
  return (uint32_t)(READ_BIT(RCC->CR, RCC_CR_MSIDIV));
}
/**
  * @}
  */

/** @defgroup RCC_LL_EF_MCO MCO
  * @{
  */

/**
  * @brief  Configure MCOx
  * @rmtoll CFGR         MCOSEL        LL_RCC_ConfigMCO\n
  *         CFGR         MCOPRE        LL_RCC_ConfigMCO
  * @param  MCOxSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MCOSOURCE_NOCLOCK
  *         @arg @ref LL_RCC_MCOSOURCE_SYSCLK
  *         @arg @ref LL_RCC_MCOSOURCE_HSI10M
  *         @arg @ref LL_RCC_MCOSOURCE_HSI
  *         @arg @ref LL_RCC_MCOSOURCE_HSE
  *         @arg @ref LL_RCC_MCOSOURCE_PLLCLK
  *         @arg @ref LL_RCC_MCOSOURCE_LSI
  *         @arg @ref LL_RCC_MCOSOURCE_LSE
  *         @arg @ref LL_RCC_MCOSOURCE_HCLK
  *         @arg @ref LL_RCC_MCOSOURCE_PCLK
  *         @arg @ref LL_RCC_MCOSOURCE_MSI
  *         @arg @ref LL_RCC_MCOSOURCE_LCD_HSI10M
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  * @param  MCOxPrescaler This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MCO_DIV_1
  *         @arg @ref LL_RCC_MCO_DIV_2
  *         @arg @ref LL_RCC_MCO_DIV_4
  *         @arg @ref LL_RCC_MCO_DIV_8
  *         @arg @ref LL_RCC_MCO_DIV_16
  *         @arg @ref LL_RCC_MCO_DIV_32
  *         @arg @ref LL_RCC_MCO_DIV_64
  *         @arg @ref LL_RCC_MCO_DIV_128
  *         @arg @ref LL_RCC_MCO_DIV_256
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ConfigMCO(uint32_t MCOxSource, uint32_t MCOxPrescaler)
{
  MODIFY_REG(RCC->CFGR, RCC_CFGR_MCOSEL | RCC_CFGR_MCOPRE, MCOxSource | MCOxPrescaler);
}

/**
  * @brief  Get MCO clock source
  * @rmtoll CFGR         MCOSEL       LL_RCC_GetMCOClockSource
  * @param  MCOx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MCO_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_MCOSOURCE_NOCLOCK
  *         @arg @ref LL_RCC_MCOSOURCE_SYSCLK
  *         @arg @ref LL_RCC_MCOSOURCE_HSI10M
  *         @arg @ref LL_RCC_MCOSOURCE_HSI
  *         @arg @ref LL_RCC_MCOSOURCE_HSE
  *         @arg @ref LL_RCC_MCOSOURCE_PLLCLK
  *         @arg @ref LL_RCC_MCOSOURCE_LSI
  *         @arg @ref LL_RCC_MCOSOURCE_LSE
  *         @arg @ref LL_RCC_MCOSOURCE_HCLK
  *         @arg @ref LL_RCC_MCOSOURCE_PCLK
  *         @arg @ref LL_RCC_MCOSOURCE_MSI
  *         @arg @ref LL_RCC_MCOSOURCE_LCD_HSI10M
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetMCOClockSource(uint32_t MCOx)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, MCOx));
}

/**
  * @brief  Get MCO division factor
  * @rmtoll CFGR         MCOPRE        LL_RCC_GetMCODiv
  * @param  MCOx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_MCO_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_MCO_DIV_1
  *         @arg @ref LL_RCC_MCO_DIV_2
  *         @arg @ref LL_RCC_MCO_DIV_4
  *         @arg @ref LL_RCC_MCO_DIV_8
  *         @arg @ref LL_RCC_MCO_DIV_16
  *         @arg @ref LL_RCC_MCO_DIV_32
  *         @arg @ref LL_RCC_MCO_DIV_64
  *         @arg @ref LL_RCC_MCO_DIV_128
  *         @arg @ref LL_RCC_MCO_DIV_256
  */
__STATIC_INLINE uint32_t LL_RCC_GetMCODiv(uint32_t MCOx)
{
  return (uint32_t)(READ_BIT(RCC->CFGR, RCC_CFGR_MCOPRE));
}

/**
  * @}
  */

/** @defgroup RCC_LL_EF_Peripheral_Clock_Source Peripheral Clock Source
  * @{
  */
#if defined(RCC_CCIPR_TIMCLKCTRL)
/**
  * @brief  Configure the TIMER PCLK frequency control.
  * @rmtoll CCIPR        TIMCLKCTRL     LL_RCC_SetTIMPCLK_MUL
  * @param  TIMPCLK_MULx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_TIMPCLK_MUL2
  *         @arg @ref LL_RCC_TIMPCLK_MUL1
  * @note   Depending on devices and packages,some TIM may not be available.
  *         Refer to device datasheet for TIM availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetTIMPCLK_MUL(uint32_t TIMPCLK_MULx)
{
  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_TIMCLKCTRL, TIMPCLK_MULx);
}

/**
  * @brief  Get the TIMPCLK multiple frequency factor.
  * @rmtoll CCIPR        TIMCLK_CTRL     LL_RCC_GetTIMPCLK_MUL
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_TIMPCLK_MUL2
  *         @arg @ref LL_RCC_TIMPCLK_MUL1
  * @note   Depending on devices and packages,some TIM may not be available.
  *         Refer to device datasheet for TIM availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetTIMPCLK_MUL(void)
{
  return (uint32_t)(READ_BIT(RCC->CCIPR, RCC_CCIPR_TIMCLKCTRL));
}
#endif /* RCC_CCIPR_TIMCLKCTRL */

#if defined(RCC_CCIPR_CANSEL)
/**
  * @brief  Configure CAN clock source
  * @param  CANSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_CAN_CLKSOURCE_PLL
  *         @arg @ref LL_RCC_CAN_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_CAN_CLKSOURCE_HSE
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetCANClockSource(uint32_t CANSource)
{
  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_CANSEL, CANSource);
}

/**
  * @brief  Get CAN clock source
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_CAN_CLKSOURCE_PLL
  *         @arg @ref LL_RCC_CAN_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_CAN_CLKSOURCE_HSE
  */
__STATIC_INLINE uint32_t LL_RCC_GetCANClockSource(void)
{
  return (uint32_t)(READ_BIT(RCC->CCIPR, RCC_CCIPR_CANSEL));
}
#endif

#if defined(RCC_CCIPR_PVDSEL)
/**
  * @brief  Configure PVD clock source
  * @rmtoll CCIPR        PVDSEL     LL_RCC_SetCOMPClockSource
  * @param  PVDSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_PVD_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_PVD_CLKSOURCE_LSC
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetPVDClockSource(uint32_t PVDSource)
{
  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_PVDSEL, PVDSource);
}

/**
  * @brief  Get PVD clock source
  * @rmtoll CCIPR        PVDSEL     LL_RCC_GetCOMPClockSource
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_PVD_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_PVD_CLKSOURCE_LSC
  */
__STATIC_INLINE uint32_t LL_RCC_GetPVDClockSource(void)
{
  return (uint32_t)(READ_BIT(RCC->CCIPR, RCC_CCIPR_PVDSEL));
}
#endif

#if defined(COMP1)
/**
  * @brief  Configure COMPx clock source
  * @rmtoll CCIPR        COMPxSEL     LL_RCC_SetCOMPClockSource
  * @param  COMPxSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE_LSC
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE_LSC
  * @note   Depending on devices and packages,some COMP may not be available.
  *         Refer to device datasheet for COMP availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetCOMPClockSource(uint32_t COMPxSource)
{
  MODIFY_REG(RCC->CCIPR, (COMPxSource & 0x0000FF00U), ((COMPxSource & 0xFFU) << 8U));
}
#endif /* COMP1 */

#if defined(COMP1)
/**
  * @brief  Get COMPx clock source
  * @rmtoll CCIPR        COMPxSEL     LL_RCC_GetCOMPClockSource
  * @param  COMPx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_COMP1_CLKSOURCE_LSC
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_COMP2_CLKSOURCE_LSC
  * @note   Depending on devices and packages,some COMP may not be available.
  *         Refer to device datasheet for COMP availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetCOMPClockSource(uint32_t COMPx)
{
  return (uint32_t)((READ_BIT(RCC->CCIPR, COMPx) >> 8U) | COMPx);
}
#endif /* COMP1 */

#if defined(RCC_CCIPR_LPUART1SEL) || (RCC_CCIPR_LPUART2SEL)
/**
  * @brief  Configure LPUARTx clock source
  * @param  LPUARTxSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE_SYSCLK
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE_SYSCLK
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE_LSE
  * @note   Depending on devices and packages,some LPUART may not be available.
  *         Refer to device datasheet for LPUART availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetLPUARTClockSource(uint32_t LPUARTxSource)
{
  MODIFY_REG(RCC->CCIPR, (LPUARTxSource & 0x0000FF00U), ((LPUARTxSource & 0xFFU) << 8U));
}

/**
  * @brief  Get LPUARTx clock source
  * @param  LPUARTx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE_SYSCLK
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LPUART1_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE_SYSCLK
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LPUART2_CLKSOURCE_LSE
  * @note   Depending on devices and packages,some LPUART may not be available.
  *         Refer to device datasheet for LPUART availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetLPUARTClockSource(uint32_t LPUARTx)
{
  return (uint32_t)((READ_BIT(RCC->CCIPR, LPUARTx) >> 8U) | LPUARTx);
}
#endif /* RCC_CCIPR_LPUART1SEL || RCC_CCIPR_LPUART2SEL */

#if defined(RCC_CCIPR_LPTIM1SEL) || (RCC_CCIPR_LPTIM2SEL)
/**
  * @brief  Configure LPTIMx clock source
  * @rmtoll CCIPR        LPTIMxSEL     LL_RCC_SetLPTIMClockSource
  * @param  LPTIMxSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LPTIM1_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_LPTIM1_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LPTIM1_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_LPTIM2_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_LPTIM2_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LPTIM2_CLKSOURCE_LSE
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetLPTIMClockSource(uint32_t LPTIMxSource)
{
  MODIFY_REG(RCC->CCIPR, (LPTIMxSource & 0xFFFF0000U), (LPTIMxSource << 16U));
}

/**
  * @brief  Get LPTIMx clock source
  * @rmtoll CCIPR        LPTIMxSEL     LL_RCC_GetLPTIMClockSource
  * @param  LPTIMx This parameter can be one of the following values:
  *         @arg @ref LL_RCC_LPTIM1_CLKSOURCE
  *         @arg @ref LL_RCC_LPTIM2_CLKSOURCE
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_LPTIM1_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_LPTIM1_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LPTIM1_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_LPTIM2_CLKSOURCE_PCLK1
  *         @arg @ref LL_RCC_LPTIM2_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_LPTIM2_CLKSOURCE_LSE
  * @note   Depending on devices and packages, some clocks may not be available.
  *         Refer to device datasheet for clocks availability.
  */
__STATIC_INLINE uint32_t LL_RCC_GetLPTIMClockSource(uint32_t LPTIMx)
{
  return (uint32_t)((READ_BIT(RCC->CCIPR, LPTIMx) >> 16U) | LPTIMx);
}
#endif /* RCC_CCIPR_LPTIM1SEL || RCC_CCIPR_LPTIM2SEL */

#if defined(RCC_CCIPR_ADCSEL)
/**
  * @brief  Configure ADC clock source
  * @param  ADCSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_ADC_CLKSOURCE_SYSCLK
  *         @arg @ref LL_RCC_ADC_CLKSOURCE_PLL
  *         @arg @ref LL_RCC_ADC_CLKSOURCE_HSI
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetADCClockSource(uint32_t ADCSource)
{
  MODIFY_REG(RCC->CCIPR, RCC_CCIPR_ADCSEL, ADCSource);
}

/**
  * @brief  Get ADC clock source
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_ADC_CLKSOURCE_SYSCLK
  *         @arg @ref LL_RCC_ADC_CLKSOURCE_PLL
  *         @arg @ref LL_RCC_ADC_CLKSOURCE_HSI
  */
__STATIC_INLINE uint32_t LL_RCC_GetADCClockSource(void)
{
  return (uint32_t)(READ_BIT(RCC->CCIPR, RCC_CCIPR_ADCSEL));
}
#endif

#if defined(RCC_BDCR_IWDGSEL)
/**
  * @brief  Configure IWDG clock source
  * @param  IWDGSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE_LSE
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetIWDGClockSource(uint32_t IWDGSource)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_IWDGSEL, IWDGSource);
}

/**
  * @brief  Get IWDG clock source
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_IWDG_CLKSOURCE_LSE
  */
__STATIC_INLINE uint32_t LL_RCC_GetIWDGClockSource(void)
{
  return (uint32_t)(READ_BIT(RCC->BDCR, RCC_BDCR_IWDGSEL));
}
#endif

/**
  * @}
  */
#if defined(RTC)
#if defined(RCC_BDCR_RTCSEL)
/** @defgroup RCC_LL_EF_RTC RTC
  * @{
  */

/**
  * @brief  Set RTC Clock Source
  * @note Once the RTC clock source has been selected, it cannot be changed anymore unless
  *       the Backup domain is reset, or unless a failure is detected on LSE (LSECSSD is
  *       set). The BDRST bit can be used to reset them.
  * @rmtoll BDCR         RTCSEL        LL_RCC_SetRTCClockSource
  * @param  Source This parameter can be one of the following values:
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_NONE
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_HSE_DIV32
  * @retval None
  */
__STATIC_INLINE void LL_RCC_SetRTCClockSource(uint32_t Source)
{
  MODIFY_REG(RCC->BDCR, RCC_BDCR_RTCSEL, Source);
}

/**
  * @brief  Get RTC Clock Source
  * @rmtoll BDCR         RTCSEL        LL_RCC_GetRTCClockSource
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_NONE
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_LSE
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_LSI
  *         @arg @ref LL_RCC_RTC_CLKSOURCE_HSE_DIV32
  */
__STATIC_INLINE uint32_t LL_RCC_GetRTCClockSource(void)
{
  return (uint32_t)(READ_BIT(RCC->BDCR, RCC_BDCR_RTCSEL));
}
#endif
/**
  * @brief  Enable RTC
  * @rmtoll BDCR         RTCEN         LL_RCC_EnableRTC
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableRTC(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_RTCEN);
}

/**
  * @brief  Disable RTC
  * @rmtoll BDCR         RTCEN         LL_RCC_DisableRTC
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableRTC(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_RTCEN);
}

/**
  * @brief  Check if RTC has been enabled or not
  * @rmtoll BDCR         RTCEN         LL_RCC_IsEnabledRTC
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledRTC(void)
{
  return ((READ_BIT(RCC->BDCR, RCC_BDCR_RTCEN) == (RCC_BDCR_RTCEN)) ? 1UL : 0UL);
}

/**
  * @brief  Force the Backup domain reset
  * @rmtoll BDCR         BDRST         LL_RCC_ForceBackupDomainReset
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ForceBackupDomainReset(void)
{
  SET_BIT(RCC->BDCR, RCC_BDCR_BDRST);
}

/**
  * @brief  Release the Backup domain reset
  * @rmtoll BDCR         BDRST         LL_RCC_ReleaseBackupDomainReset
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ReleaseBackupDomainReset(void)
{
  CLEAR_BIT(RCC->BDCR, RCC_BDCR_BDRST);
}

/**
  * @}
  */
#endif

#if defined(RCC_PLL_SUPPORT)
/** @defgroup RCC_LL_EF_PLL PLL
  * @{
  */

/**
  * @brief  Enable PLL
  * @rmtoll CR           PLLON         LL_RCC_PLL_Enable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_PLL_Enable(void)
{
  SET_BIT(RCC->CR, RCC_CR_PLLON);
}

/**
  * @brief  Disable PLL
  * @note Cannot be disabled if the PLL clock is used as the system clock
  * @rmtoll CR           PLLON         LL_RCC_PLL_Disable
  * @retval None
  */
__STATIC_INLINE void LL_RCC_PLL_Disable(void)
{
  CLEAR_BIT(RCC->CR, RCC_CR_PLLON);
}

/**
  * @brief  Check if PLL Ready
  * @rmtoll CR           PLLRDY        LL_RCC_PLL_IsReady
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_PLL_IsReady(void)
{
  return ((READ_BIT(RCC->CR, RCC_CR_PLLRDY) == (RCC_CR_PLLRDY)) ? 1UL : 0UL);
}

/**
  * @brief  Configure PLL used for SYSCLK Domain
  * @rmtoll PLLCFGR      PLLSRC        LL_RCC_PLL_ConfigDomain_SYS\n
  *         PLLCFGR      PLLMUL        LL_RCC_PLL_ConfigDomain_SYS\n
  * @param  Source This parameter can be one of the following values:
  *         @arg @ref LL_RCC_PLLSOURCE_HSI
  *         @arg @ref LL_RCC_PLLSOURCE_HSE
  * @param  PLLMul This parameter can be one of the following values:
  *         @arg @ref LL_RCC_PLL_MUL2 
  *         @arg @ref LL_RCC_PLL_MUL3 
  *         @arg @ref LL_RCC_PLL_MUL4 
  *         @arg @ref LL_RCC_PLL_MUL5 
  *         @arg @ref LL_RCC_PLL_MUL6 
  *         @arg @ref LL_RCC_PLL_MUL7 
  *         @arg @ref LL_RCC_PLL_MUL8 
  *         @arg @ref LL_RCC_PLL_MUL9 
  *         @arg @ref LL_RCC_PLL_MUL10
  *         @arg @ref LL_RCC_PLL_MUL11
  *         @arg @ref LL_RCC_PLL_MUL12
  *         @arg @ref LL_RCC_PLL_MUL13
  *         @arg @ref LL_RCC_PLL_MUL14
  *         @arg @ref LL_RCC_PLL_MUL15
  *         @arg @ref LL_RCC_PLL_MUL16
  *         @arg @ref LL_RCC_PLL_MUL17
  *         @arg @ref LL_RCC_PLL_MUL18
  * @retval None
  */
__STATIC_INLINE void LL_RCC_PLL_ConfigDomain_SYS(uint32_t Source, uint32_t PLLMul)
{
  MODIFY_REG(RCC->PLLCFGR, RCC_PLLCFGR_PLLSRC | RCC_PLLCFGR_PLLMUL, (Source & RCC_PLLCFGR_PLLSRC) | PLLMul);
}

/**
  * @brief  Configure PLL clock source
  * @rmtoll PLLCFGR      PLLSRC        LL_RCC_PLL_SetMainSource
  * @param PLLSource This parameter can be one of the following values:
  *         @arg @ref LL_RCC_PLLSOURCE_HSI
  *         @arg @ref LL_RCC_PLLSOURCE_HSE
  * @retval None
  */
__STATIC_INLINE void LL_RCC_PLL_SetMainSource(uint32_t PLLSource)
{
  MODIFY_REG(RCC->PLLCFGR, RCC_PLLCFGR_PLLSRC, PLLSource);
}

/**
  * @brief  Get the oscillator used as PLL clock source.
  * @rmtoll PLLCFGR      PLLSRC        LL_RCC_PLL_GetMainSource
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_PLLSOURCE_HSI
  *         @arg @ref LL_RCC_PLLSOURCE_HSE
  */
__STATIC_INLINE uint32_t LL_RCC_PLL_GetMainSource(void)
{
  return (uint32_t)(READ_BIT(RCC->PLLCFGR, RCC_PLLCFGR_PLLSRC));
}

/**
  * @brief  Configure PLL Multiplication Factor.
  * @param PLLMul This parameter can be one of the following values:
  *         @arg @ref LL_RCC_PLL_MUL2 
  *         @arg @ref LL_RCC_PLL_MUL3 
  *         @arg @ref LL_RCC_PLL_MUL4 
  *         @arg @ref LL_RCC_PLL_MUL5 
  *         @arg @ref LL_RCC_PLL_MUL6 
  *         @arg @ref LL_RCC_PLL_MUL7 
  *         @arg @ref LL_RCC_PLL_MUL8 
  *         @arg @ref LL_RCC_PLL_MUL9 
  *         @arg @ref LL_RCC_PLL_MUL10
  *         @arg @ref LL_RCC_PLL_MUL11
  *         @arg @ref LL_RCC_PLL_MUL12
  *         @arg @ref LL_RCC_PLL_MUL13
  *         @arg @ref LL_RCC_PLL_MUL14
  *         @arg @ref LL_RCC_PLL_MUL15
  *         @arg @ref LL_RCC_PLL_MUL16
  *         @arg @ref LL_RCC_PLL_MUL17
  *         @arg @ref LL_RCC_PLL_MUL18
  * @retval None
  */
__STATIC_INLINE void LL_RCC_PLL_SetMulFactor(uint32_t PLLMul)
{
  MODIFY_REG(RCC->PLLCFGR, RCC_PLLCFGR_PLLMUL, PLLMul);
}

/**
  * @brief  Get PLL multiplication Factor
  * @rmtoll PLLCFGR      PLLMUL        LL_RCC_PLL_GetMultiplicator
  * @retval Returned value can be one of the following values:
  *         @arg @ref LL_RCC_PLL_MUL2 
  *         @arg @ref LL_RCC_PLL_MUL3 
  *         @arg @ref LL_RCC_PLL_MUL4 
  *         @arg @ref LL_RCC_PLL_MUL5 
  *         @arg @ref LL_RCC_PLL_MUL6 
  *         @arg @ref LL_RCC_PLL_MUL7 
  *         @arg @ref LL_RCC_PLL_MUL8 
  *         @arg @ref LL_RCC_PLL_MUL9 
  *         @arg @ref LL_RCC_PLL_MUL10
  *         @arg @ref LL_RCC_PLL_MUL11
  *         @arg @ref LL_RCC_PLL_MUL12
  *         @arg @ref LL_RCC_PLL_MUL13
  *         @arg @ref LL_RCC_PLL_MUL14
  *         @arg @ref LL_RCC_PLL_MUL15
  *         @arg @ref LL_RCC_PLL_MUL16
  *         @arg @ref LL_RCC_PLL_MUL17
  *         @arg @ref LL_RCC_PLL_MUL18
  */
__STATIC_INLINE uint32_t LL_RCC_PLL_GetMultiplicator(void)
{
  return (uint32_t)(READ_BIT(RCC->PLLCFGR, RCC_PLLCFGR_PLLMUL));
}
/**
  * @}
  */
#endif


/** @defgroup RCC_LL_EF_FLAG_Management FLAG Management
  * @{
  */

/**
  * @brief  Clear LSI ready interrupt flag
  * @rmtoll CICR         LSIRDYC       LL_RCC_ClearFlag_LSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_LSIRDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_LSIRDYC);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Clear LSE ready interrupt flag
  * @rmtoll CICR         LSERDYC       LL_RCC_ClearFlag_LSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_LSERDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_LSERDYC);
}
#endif

/**
  * @brief  Clear MSI ready interrupt flag
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_MSIRDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_MSIRDYC);
}

/**
  * @brief  Clear HSI ready interrupt flag
  * @rmtoll CICR         HSIRDYC       LL_RCC_ClearFlag_HSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_HSIRDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_HSIRDYC);
}

#if defined(RCC_HSE_SUPPORT)
/**
  * @brief  Clear HSE ready interrupt flag
  * @rmtoll CICR         HSERDYC       LL_RCC_ClearFlag_HSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_HSERDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_HSERDYC);
}
#endif

#if defined(RCC_PLL_SUPPORT)
/**
  * @brief  Clear PLL ready interrupt flag
  * @rmtoll CICR         PLLRDYC       LL_RCC_ClearFlag_PLLRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_PLLRDY(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_PLLRDYC);
}
#endif
#if defined(RCC_HSE_SUPPORT)
/**
  * @brief  Clear Clock security system interrupt flag
  * @rmtoll CICR         CSSC          LL_RCC_ClearFlag_HSECSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_HSECSS(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_CSSC);
}
#endif
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Clear LSE Clock security system interrupt flag
  * @rmtoll CICR         LSECSSC       LL_RCC_ClearFlag_LSECSS
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearFlag_LSECSS(void)
{
  SET_BIT(RCC->CICR, RCC_CICR_LSECSSC);
}
#endif

/**
  * @brief  Check if LSI ready interrupt occurred or not
  * @rmtoll CIFR         LSIRDYF       LL_RCC_IsActiveFlag_LSIRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_LSIRDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_LSIRDYF) == (RCC_CIFR_LSIRDYF)) ? 1UL : 0UL);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Check if LSE ready interrupt occurred or not
  * @rmtoll CIFR         LSERDYF       LL_RCC_IsActiveFlag_LSERDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_LSERDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_LSERDYF) == (RCC_CIFR_LSERDYF)) ? 1UL : 0UL);
}
#endif

/**
  * @brief  Check if MSI ready interrupt occurred or not
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_MSIRDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_MSIRDYF) == (RCC_CIFR_MSIRDYF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if HSI ready interrupt occurred or not
  * @rmtoll CIFR         HSIRDYF       LL_RCC_IsActiveFlag_HSIRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_HSIRDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_HSIRDYF) == (RCC_CIFR_HSIRDYF)) ? 1UL : 0UL);
}
#if defined(RCC_HSE_SUPPORT)
/**
  * @brief  Check if HSE ready interrupt occurred or not
  * @rmtoll CIFR         HSERDYF       LL_RCC_IsActiveFlag_HSERDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_HSERDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_HSERDYF) == (RCC_CIFR_HSERDYF)) ? 1UL : 0UL);
}
#endif
#if defined(RCC_PLL_SUPPORT)
/**
  * @brief  Check if PLL ready interrupt occurred or not
  * @rmtoll CIFR         PLLRDYF       LL_RCC_IsActiveFlag_PLLRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_PLLRDY(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_PLLRDYF) == (RCC_CIFR_PLLRDYF)) ? 1UL : 0UL);
}
#endif
#if defined(RCC_HSE_SUPPORT)
/**
  * @brief  Check if Clock security system interrupt occurred or not
  * @rmtoll CIFR         CSSF          LL_RCC_IsActiveFlag_HSECSS
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_HSECSS(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_CSSF) == (RCC_CIFR_CSSF)) ? 1UL : 0UL);
}
#endif
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Check if LSE Clock security system interrupt occurred or not
  * @rmtoll CIFR         LSECSSF       LL_RCC_IsActiveFlag_LSECSS
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_LSECSS(void)
{
  return ((READ_BIT(RCC->CIFR, RCC_CIFR_LSECSSF) == (RCC_CIFR_LSECSSF)) ? 1UL : 0UL);
}
#endif

/**
  * @brief  Check if RCC flag Independent Watchdog reset is set or not.
  * @rmtoll CSR          IWDGRSTF      LL_RCC_IsActiveFlag_IWDGRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_IWDGRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_IWDGRSTF) == (RCC_CSR_IWDGRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag Option byte reset is set or not.
  * @rmtoll CSR          OBLRSTF       LL_RCC_IsActiveFlag_OBLRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_OBLRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_OBLRSTF) == (RCC_CSR_OBLRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag Pin reset is set or not.
  * @rmtoll CSR          PINRSTF       LL_RCC_IsActiveFlag_PINRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_PINRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_PINRSTF) == (RCC_CSR_PINRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag Software reset is set or not.
  * @rmtoll CSR          SFTRSTF       LL_RCC_IsActiveFlag_SFTRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_SFTRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_SFTRSTF) == (RCC_CSR_SFTRSTF)) ? 1UL : 0UL);
}

#if defined(WWDG)
/**
  * @brief  Check if RCC flag Window Watchdog reset is set or not.
  * @rmtoll CSR          WWDGRSTF      LL_RCC_IsActiveFlag_WWDGRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_WWDGRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_WWDGRSTF) == (RCC_CSR_WWDGRSTF)) ? 1UL : 0UL);
}
#endif

/**
  * @brief  Check if RCC flag BOR or POR/PDR reset is set or not.
  * @rmtoll CSR          PWRRSTF       LL_RCC_IsActiveFlag_PWRRST
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_PWRRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_PWRRSTF) == (RCC_CSR_PWRRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag SRAM1/SRAM2 parity error reset is set or not.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_SPERST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_SPERSTF) == (RCC_CSR_SPERSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Check if RCC flag Stop and Standby Low Power Mode Reset is set or not.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsActiveFlag_LPWRRST(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_LPWRRSTF) == (RCC_CSR_LPWRRSTF)) ? 1UL : 0UL);
}

/**
  * @brief  Set RMVF bit to clear the reset flags.
  * @rmtoll CSR          RMVF          LL_RCC_ClearResetFlags
  * @retval None
  */
__STATIC_INLINE void LL_RCC_ClearResetFlags(void)
{
  SET_BIT(RCC->CSR, RCC_CSR_RMVF);
}

/**
  * @brief  Enable NRST filter
  * @rmtoll CSR          NRST_FLTDIS         LL_RCC_EnableNRSTFilter
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableNRSTFilter(void)
{
  CLEAR_BIT(RCC->CSR, RCC_CSR_NRST_FLTDIS);
}

/**
  * @brief  Disable NRST filter
  * @rmtoll CSR          NRST_FLTDIS         LL_RCC_DisableNRSTFilter
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableNRSTFilter(void)
{
  SET_BIT(RCC->CSR, RCC_CSR_NRST_FLTDIS);
}

/**
  * @brief  Check if NRST filter is enable
  * @rmtoll CSR          NRST_FLTDIS         LL_RCC_IsEnableNRSTFilter
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnableNRSTFilter(void)
{
  return ((READ_BIT(RCC->CSR, RCC_CSR_NRST_FLTDIS) == (RCC_CSR_NRST_FLTDIS)) ? 0UL : 1UL);
}

/**
  * @}
  */

/** @defgroup RCC_LL_EF_IT_Management IT Management
  * @{
  */

/**
  * @brief  Enable LSI ready interrupt
  * @rmtoll CIER         LSIRDYIE      LL_RCC_EnableIT_LSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_LSIRDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_LSIRDYIE);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Enable LSE ready interrupt
  * @rmtoll CIER         LSERDYIE      LL_RCC_EnableIT_LSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_LSERDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_LSERDYIE);
}
#endif

/**
  * @brief  Enable MSI ready interrupt
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_MSIRDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_MSIRDYIE);
}

/**
  * @brief  Enable HSI ready interrupt
  * @rmtoll CIER         HSIRDYIE      LL_RCC_EnableIT_HSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_HSIRDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_HSIRDYIE);
}
#if defined(RCC_HSE_SUPPORT)
/**
  * @brief  Enable HSE ready interrupt
  * @rmtoll CIER         HSERDYIE      LL_RCC_EnableIT_HSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_HSERDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_HSERDYIE);
}
#endif
#if defined(RCC_PLL_SUPPORT)
/**
  * @brief  Enable PLL ready interrupt
  * @rmtoll CIER         PLLRDYIE      LL_RCC_EnableIT_PLLRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_EnableIT_PLLRDY(void)
{
  SET_BIT(RCC->CIER, RCC_CIER_PLLRDYIE);
}
#endif
/**
  * @brief  Disable LSI ready interrupt
  * @rmtoll CIER         LSIRDYIE      LL_RCC_DisableIT_LSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_LSIRDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_LSIRDYIE);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Disable LSE ready interrupt
  * @rmtoll CIER         LSERDYIE      LL_RCC_DisableIT_LSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_LSERDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_LSERDYIE);
}
#endif

/**
  * @brief  Disable MSI ready interrupt
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_MSIRDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_MSIRDYIE);
}

/**
  * @brief  Disable HSI ready interrupt
  * @rmtoll CIER         HSIRDYIE      LL_RCC_DisableIT_HSIRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_HSIRDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_HSIRDYIE);
}
#if defined(RCC_HSE_SUPPORT)
/**
  * @brief  Disable HSE ready interrupt
  * @rmtoll CIER         HSERDYIE      LL_RCC_DisableIT_HSERDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_HSERDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_HSERDYIE);
}
#endif
#if defined(RCC_PLL_SUPPORT)
/**
  * @brief  Disable PLL ready interrupt
  * @rmtoll CIER         PLLRDYIE      LL_RCC_DisableIT_PLLRDY
  * @retval None
  */
__STATIC_INLINE void LL_RCC_DisableIT_PLLRDY(void)
{
  CLEAR_BIT(RCC->CIER, RCC_CIER_PLLRDYIE);
}
#endif
/**
  * @brief  Checks if LSI ready interrupt source is enabled or disabled.
  * @rmtoll CIER         LSIRDYIE      LL_RCC_IsEnabledIT_LSIRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_LSIRDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_LSIRDYIE) == (RCC_CIER_LSIRDYIE)) ? 1UL : 0UL);
}
#if defined(RCC_LSE_SUPPORT)
/**
  * @brief  Checks if LSE ready interrupt source is enabled or disabled.
  * @rmtoll CIER         LSERDYIE      LL_RCC_IsEnabledIT_LSERDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_LSERDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_LSERDYIE) == (RCC_CIER_LSERDYIE)) ? 1UL : 0UL);
}
#endif

/**
  * @brief  Checks if MSI ready interrupt source is enabled or disabled.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_MSIRDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_MSIRDYIE) == (RCC_CIER_MSIRDYIE)) ? 1UL : 0UL);
}

/**
  * @brief  Checks if HSI ready interrupt source is enabled or disabled.
  * @rmtoll CIER         HSIRDYIE      LL_RCC_IsEnabledIT_HSIRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_HSIRDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_HSIRDYIE) == (RCC_CIER_HSIRDYIE)) ? 1UL : 0UL);
}

#if defined(RCC_HSE_SUPPORT)
/**
  * @brief  Checks if HSE ready interrupt source is enabled or disabled.
  * @rmtoll CIER         HSERDYIE      LL_RCC_IsEnabledIT_HSERDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_HSERDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_HSERDYIE) == (RCC_CIER_HSERDYIE)) ? 1UL : 0UL);
}
#endif
#if defined(RCC_PLL_SUPPORT)
/**
  * @brief  Checks if PLL ready interrupt source is enabled or disabled.
  * @rmtoll CIER         PLLRDYIE      LL_RCC_IsEnabledIT_PLLRDY
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t LL_RCC_IsEnabledIT_PLLRDY(void)
{
  return ((READ_BIT(RCC->CIER, RCC_CIER_PLLRDYIE) == (RCC_CIER_PLLRDYIE)) ? 1UL : 0UL);
}
#endif
/**
  * @}
  */

#if defined(USE_FULL_LL_DRIVER)
/** @defgroup RCC_LL_EF_Init De-initialization function
  * @{
  */
ErrorStatus LL_RCC_DeInit(void);
/**
  * @}
  */

/** @defgroup RCC_LL_EF_Get_Freq Get system and peripherals clocks frequency functions
  * @{
  */
void        LL_RCC_GetSystemClocksFreq(LL_RCC_ClocksTypeDef *RCC_Clocks);
uint32_t    LL_RCC_GetMCOClockFreq(uint32_t MCOx);
uint32_t    LL_RCC_GetLSCClockFreq(void);
#if defined(RCC_CCIPR_PVDSEL)
uint32_t    LL_RCC_GetPVDClockFreq(void);
#endif
#if defined(COMP1) || defined (COMP2)
uint32_t    LL_RCC_GetCOMPClockFreq(uint32_t COMPx);
#endif
#if defined(LPTIM1) || defined (LPTIM2)
uint32_t    LL_RCC_GetLPTIMClockFreq(uint32_t LPTIMx);
#endif 
#if defined(RTC)
uint32_t    LL_RCC_GetRTCClockFreq(void);
#endif
#if defined(LPUART1) || defined (LPUART2)
uint32_t    LL_RCC_GetLPUARTClockFreq(uint32_t LPUARTx);
#endif
/**
  * @}
  */
#endif /* USE_FULL_LL_DRIVER */

/**
  * @}
  */

/**
  * @}
  */

#endif /* defined(RCC) */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __PY32T092_LL_RCC_H */

/************************ (C) COPYRIGHT Puya *****END OF FILE****/
