/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private define ------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
uint32_t LED_BlinkingTime = 0;
GPIO_InitTypeDef  GPIO_InitStruct = {0};
/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
  
/**
  * @brief  Main program.
  * @retval int
  */
int main(void)
{
  /* Reset of all peripherals, Initializes the Systick. */
  HAL_Init();
  
  /* System clock configuration */
  APP_SystemClockConfig(); 

  /* Initialize LED */
  BSP_LED_Init(LED_GREEN);

  /* Initialization button */
  BSP_PB_Init(BUTTON_KEY,BUTTON_MODE_GPIO);
  
  /* Turn on the LED */
  BSP_LED_On(LED_GREEN);
  
  if (__HAL_PWR_GET_FLAG(PWR_FLAG_SB) != RESET)
  {
    /* Clear the STANDBY flag */
    __HAL_PWR_CLEAR_FLAG(PWR_FLAG_SB);

    /* Clear the wake-up flag */
    __HAL_PWR_CLEAR_FLAG(PWR_FLAG_WU);
    
    LED_BlinkingTime=0;
    
    while(LED_BlinkingTime<10)
    {
      HAL_Delay(500);
      BSP_LED_Toggle(LED_GREEN);
      LED_BlinkingTime++;
    }    
  }
  /* Enable pull-up and pull-down configuration */
  HAL_PWREx_EnablePullUpPullDownConfig();
  
  /* Before entering standby mode, the pull-up function needs to be enabled 
     when P13 is used as the NRST_Pin, whereas it does not need to be enabled 
     when PD13 is used as a GPIO */
  HAL_PWREx_EnableGPIOPullUp(PWR_GPIO_D, PWR_GPIO_BIT_13);
  
  /* The Following Wakeup sequence is highly recommended prior to each Standby mode entry
     mainly when using more than one wakeup source this is to not miss any wakeup event.
     - Configure GPIO,
     - Disable all used wakeup sources,
     - Clear all related wakeup flags,
     - Re-enable all used wakeup sources,
     - Enter the Standby mode.
  */
  /* Configure PA1 as input mode */
  __HAL_RCC_GPIOA_CLK_ENABLE();
  GPIO_InitStruct.Mode   = GPIO_MODE_INPUT;
  GPIO_InitStruct.Pull   = GPIO_PULLDOWN;
  GPIO_InitStruct.Speed  = GPIO_SPEED_FREQ_VERY_HIGH;
  GPIO_InitStruct.Pin    = GPIO_PIN_1;
  HAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
  
  /* Disable all used wakeup sources: PWR_WAKEUP_PIN4 */
  HAL_PWR_DisableWakeUpPin(PWR_WAKEUP_PIN4);
  
  /* Clear the wake-up flag */
  __HAL_PWR_CLEAR_FLAG(PWR_FLAG_WU);
  
  /* Enable WakeUp Pin PWR_WAKEUP_PIN4 connected to PA1 */
  HAL_PWR_EnableWakeUpPin(PWR_WAKEUP_PIN4_HIGH);

  /* Wait for the button to be pressed */
  while(BSP_PB_GetState(BUTTON_KEY) == 0);

  /* Turn off the LED */
  BSP_LED_Off(LED_GREEN);
  
  /* Disable SYSTICK interrupt to prevent SYSTICK from waking up */
  HAL_SuspendTick();

  /* Enter STANDBY mode */
  HAL_PWR_EnterSTANDBYMode(PWR_DEEPLOWPOWERREGULATOR_ON);
  
  /* Resume the SysTick interrupt */
  HAL_ResumeTick();
  
  /* infinite loop */
  while (1)
  {
  }
}

/**
  * @brief  System clock configuration function
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

  /* Oscillator configuration */
  RCC_OscInitStruct.OscillatorType = RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_LSI | RCC_OSCILLATORTYPE_MSI; /* Select oscillator HSI, LSI, MSI */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType |= RCC_OSCILLATORTYPE_HSE; /* Select oscillator HSE */
#endif /* RCC_HSE_SUPPORT */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType |= RCC_OSCILLATORTYPE_LSE; /* Select oscillator LSE */
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.HSIState = RCC_HSI_ON;                          /* Enable HSI */
  RCC_OscInitStruct.HSIDiv = RCC_HSI_DIV1;                          /* HSI 1 frequency division */
  RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_8MHz;  /* Configure HSI clock 8MHz */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.HSEState = RCC_HSE_OFF;                         /* Close HSE */
  /*RCC_OscInitStruct.HSEFreq = RCC_HSE_24_32MHz;*/
#endif /* RCC_HSE_SUPPORT */
  RCC_OscInitStruct.LSIState = RCC_LSI_OFF;                         /* Close LSI */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.LSEState = RCC_LSE_OFF;                         /* Close LSE */
  /*RCC_OscInitStruct.LSEDriver = RCC_LSEDRIVE_MEDIUM;*/
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.MSIState = RCC_MSI_OFF;                         /* Close MSI */
  /*RCC_OscInitStruct.MSIDiv = RCC_MSI_DIV1;*/
  RCC_OscInitStruct.PLL.PLLState = RCC_PLL_OFF;                     /* Close PLL */
  /*RCC_OscInitStruct.PLL.PLLSource = RCC_PLLSOURCE_HSI;*/
  /*RCC_OscInitStruct.PLL.PLLMUL = RCC_PLL_MUL2;*/
  /* Configure oscillator */
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    APP_ErrorHandler();
  }

  /* Clock source configuration */
  RCC_ClkInitStruct.ClockType = RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_PCLK; /* Choose to configure clock HCLK, SYSCLK, PCLK */
  RCC_ClkInitStruct.SYSCLKSource = RCC_SYSCLKSOURCE_HSISYS; /* Select HSISYS as the system clock */
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;        /* AHB clock 1 division */
  RCC_ClkInitStruct.APBCLKDivider = RCC_HCLK_DIV1;          /* APB clock 1 division */
  /* Configure clock source */
  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_0) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
