/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private define ------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
RTC_HandleTypeDef RtcHandle;
RTC_TimeTypeDef   Rtctime;
RTC_AlarmTypeDef  Rtc_AlarmStruct;

/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
static void APP_RtcHandle(void);
static void APP_RtcSetAlarm(void);

/**
  * @brief  Main program.
  * @retval int
  */
int main(void)
{
  /* Reset of all peripherals, Initializes the Systick */
  HAL_Init();
  
  /* Enable LSI clock */
  APP_SystemClockConfig();
  
  /* lnitialize LED */
  BSP_LED_Init(LED_GREEN);
  
  /* Initialize button */
  BSP_PB_Init(BUTTON_USER, BUTTON_MODE_GPIO);
  
  /* Configure UART */
  BSP_UART_Config();

  /* RTC initialization */
  APP_RtcHandle();
  
  /* Turn on LED */
  BSP_LED_On(LED_GREEN);
  
  /* Wait for button press */
  while (BSP_PB_GetState(BUTTON_USER) == 0)
  {
  }
  
  /* Get current RTC time in binary format */
  HAL_RTC_GetTime(&RtcHandle,&Rtctime,RTC_FORMAT_BIN);
  
  /* Configure the time of the first alarm in binary format */
  Rtc_AlarmStruct.AlarmTime         = Rtctime;
  Rtc_AlarmStruct.AlarmTime.Seconds += 1;
  HAL_RTC_SetAlarm_IT(&RtcHandle, &Rtc_AlarmStruct, RTC_FORMAT_BIN);

   /* Turn off LED */
  BSP_LED_Off(LED_GREEN);

  /* infinite loop */
  while (1)
  {
    /* Suspend systick interrupt */
    HAL_SuspendTick();

    /* Entering STOP mode */
    HAL_PWR_EnterSTOPMode(PWR_DEEPLOWPOWERREGULATOR_ON, PWR_STOPENTRY_WFI);

    /* Resume systick interrupt */
    HAL_ResumeTick();

    /* Wait for synchronization */
    HAL_RTC_WaitForSynchro(&RtcHandle);
    
    /* Set RTC alarm */
    APP_RtcSetAlarm();
  }
}

/**
  * @brief  RTC initialization function, sets RTC to January 1, 2025, Saturday, 00:00:00
  * @param  None
  * @retval None
  */
static void APP_RtcHandle(void)
{
  RTC_TimeTypeDef Timeinit = {0};
  
  /* Enable access to the backup domain (RTC settings are stored in the backup domain) */
  HAL_PWR_EnableBkUpAccess();

  /* RTC initialization */
  RtcHandle.Instance             = RTC;                       /* Select RTC */
  RtcHandle.Init.AsynchPrediv    = RTC_AUTO_1_SECOND;         /* Automatic calculation of RTC's 1-second time base */
  RtcHandle.Init.OutPut          = RTC_OUTPUTSOURCE_NONE;     /* No output on the TAMPER pin */
  /*2025-1-1-00:00:00*/
  RtcHandle.DateToUpdate.Year    = 25;                        /* Year 25   */
  RtcHandle.DateToUpdate.Month   = RTC_MONTH_JANUARY;         /* January   */
  RtcHandle.DateToUpdate.Date    = 1;                         /* 1st day   */
  RtcHandle.DateToUpdate.WeekDay = RTC_WEEKDAY_SATURDAY;      /* Saturday  */
  Timeinit.Hours                 = 0x00;                      /* 0 hours   */
  Timeinit.Minutes               = 0x00;                      /* 0 minutes */
  Timeinit.Seconds               = 0x00;                      /* 0 seconds */

  /* RTC initialization */
  HAL_RTC_Init(&RtcHandle);

  /* Set RTC current time in binary format */
  HAL_RTC_SetTime(&RtcHandle, &Timeinit, RTC_FORMAT_BIN);
}

/**
  * @brief  Set RTC alarm
  * @param  None
  * @retval None
  */
static void APP_RtcSetAlarm(void)
{
  /* Get current alarm time in binary format */
  HAL_RTC_GetAlarm(&RtcHandle, &Rtc_AlarmStruct, RTC_FORMAT_BIN);
  
  /* Update alarm time in binary format */
  Rtc_AlarmStruct.AlarmTime.Seconds += 1;
  HAL_RTC_SetAlarm_IT(&RtcHandle, &Rtc_AlarmStruct, RTC_FORMAT_BIN);
}

/**
  * @brief  RTC alarm event callback function, LED turns on when exiting low-power mode
  * @param  hrtc：RTC handle
  * @retval None
  */
void HAL_RTC_AlarmAEventCallback(RTC_HandleTypeDef *hrtc)
{
  /* Toggle LED */
  BSP_LED_Toggle(LED_GREEN);
  
  /* Print the current alarm time */
  printf("%02d:%02d:%02d\r\n", Rtc_AlarmStruct.AlarmTime.Hours , Rtc_AlarmStruct.AlarmTime.Minutes, Rtc_AlarmStruct.AlarmTime.Seconds);
}

/**
  * @brief  System clock configuration function
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  RCC_OscInitTypeDef RCC_OscInitStruct = {0};
  RCC_ClkInitTypeDef RCC_ClkInitStruct = {0};

  /* Oscillator configuration */
  RCC_OscInitStruct.OscillatorType      = RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_LSI; /* Select oscillator HSI, LSI */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType      |= RCC_OSCILLATORTYPE_HSE; /* Select oscillator HSE */
#endif /* RCC_HSE_SUPPORT */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.OscillatorType      |= RCC_OSCILLATORTYPE_LSE; /* Select oscillator LSE */
#endif /* RCC_LSE_SUPPORT */
  RCC_OscInitStruct.HSIState            = RCC_HSI_ON;              /* Enable HSI */
  RCC_OscInitStruct.HSIDiv              = RCC_HSI_DIV1;            /* HSI 1 frequency division */
  RCC_OscInitStruct.HSICalibrationValue = RCC_HSICALIBRATION_8MHz; /* Configure HSI clock 8MHz */
#if defined(RCC_HSE_SUPPORT)
  RCC_OscInitStruct.HSEState            = RCC_HSE_OFF;             /* Close HSE */
  /*RCC_OscInitStruct.HSEFreq             = RCC_HSE_24_32MHz;*/
#endif /* RCC_HSE_SUPPORT */
  RCC_OscInitStruct.LSIState            = RCC_LSI_OFF;             /* Close LSI */
#if defined(RCC_LSE_SUPPORT)
  RCC_OscInitStruct.LSEState            = RCC_LSE_OFF;             /* Close LSE */
  /*RCC_OscInitStruct.LSEDriver           = RCC_LSEDRIVE_MEDIUM;*/
#endif /* RCC_LSE_SUPPORT */
#if defined(RCC_PLL_SUPPORT)
  RCC_OscInitStruct.PLL.PLLState        = RCC_PLL_OFF;             /* Close PLL */
  /*RCC_OscInitStruct.PLL.PLLSource       = RCC_PLLSOURCE_HSI;*/
  /*RCC_OscInitStruct.PLL.PLLMUL          = RCC_PLL_MUL6;*/
  /*RCC_OscInitStruct.PLL.PLLDiv          = RCC_PLL_DIV1;*/
#endif /* RCC_PLL_SUPPORT */
  /* Configure oscillator */
  if (HAL_RCC_OscConfig(&RCC_OscInitStruct) != HAL_OK)
  {
    APP_ErrorHandler();
  }

  /* Clock source configuration */
  RCC_ClkInitStruct.ClockType     = RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_PCLK; /* Choose to configure clock HCLK, SYSCLK, PCLK */
  RCC_ClkInitStruct.SYSCLKSource  = RCC_SYSCLKSOURCE_HSISYS; /* Select HSISYS as the system clock */
  RCC_ClkInitStruct.AHBCLKDivider = RCC_SYSCLK_DIV1;         /* AHB clock 1 division */
  RCC_ClkInitStruct.APBCLKDivider = RCC_HCLK_DIV1;           /* APB clock 1 division */
  /* Configure clock source */
  if (HAL_RCC_ClockConfig(&RCC_ClkInitStruct, FLASH_LATENCY_0) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  Error executing function.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* Users can add their own printing information as needed,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line) */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
