/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2023 Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by Puya under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Private define ------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
USBH_HandleTypeDef          hUSBH;
HID_ApplicationTypeDef      App_State = APPLICATION_IDLE;
FATFS                       USBDISKFatFs;
__IO MSC_APP_State          msc_app_state = MSC_APP_WAIT;

/* Private user code ---------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
static void APP_SystemClockConfig(void);
static void USBH_UserProcess(USBH_HandleTypeDef *phost, uint8_t id);
static void USB_demo(USBH_HandleTypeDef *phost);

/**
  * @brief  Main program.
  * @retval int
  */
int main(void)
{
  /* Reset of all peripherals, Initializes the Systick. */
  HAL_Init();
  
  /* Configure system clock */
  APP_SystemClockConfig(); 

  /* Initialize LED */
  BSP_LED_Init(LED_GREEN);

  /* Initialize debug UART (used for printf) */
  BSP_UART_Config();
  
  /* Init Host Library */
  USBH_Init(&hUSBH, USBH_UserProcess, 1);
  
  /* Add Supported Class */
  USBH_RegisterClass(&hUSBH, USBH_MSC_CLASS);
  
  /* Start Host Process */
  USBH_Start(&hUSBH);

  /* Infinite loop */
  while (1)
  {
    /* USB Host Background task */
    USBH_Process(&hUSBH);
    
    /* USB user task */
    USB_demo(&hUSBH);

  }
}

/**
  * @brief  User USB host demo.
  * @param  phost: USB HOST handle
  * @retval None
  */
static void USB_demo(USBH_HandleTypeDef *phost)
{
  switch(msc_app_state)
  {
    case MSC_APP_WAIT:
      msc_app_state = MSC_APP_FILE_OPERATIONS;
    break;
    
    case MSC_APP_FILE_OPERATIONS:
      /* Read and Write File Here */
      if(App_State == APPLICATION_READY)
      {
        USBH_UsrLog("\n\n*** Files operations ***\n");
        if(msc_file_operations() != 0)
        {
          USBH_UsrLog("\n\n*** Files operations error ***\n");
        }
        msc_app_state = MSC_APP_IDLE;
      }
    break;

    default:
    break;
  }
  
  if(App_State == APPLICATION_DISCONNECT)
  {
    App_State = APPLICATION_IDLE;
    USBH_UsrLog("USB device disconnected !!! \n");
    msc_app_state = MSC_APP_WAIT;
  }
}

/**
  * @brief  User Process
  * @param  phost: Host Handle
  * @param  id: Host Library user message ID
  * @retval None
  */
static void USBH_UserProcess(USBH_HandleTypeDef *phost, uint8_t id)
{
  switch (id)
  {
    case HOST_USER_SELECT_CONFIGURATION:
      break;

    case HOST_USER_DISCONNECTION:
      USBH_UsrLog("Disconnect\r\n");   
      App_State = APPLICATION_DISCONNECT;
      if (f_mount(NULL, "", 0) != FR_OK)
      {
        USBH_UsrLog("ERROR : Cannot DeInitialize FatFs! \n");
      }
      break;

    case HOST_USER_CLASS_ACTIVE:
      App_State = APPLICATION_READY;
      if (f_mount(&USBDISKFatFs, "2:", 0)!= FR_OK)
      {
        USBH_UsrLog("ERROR : Cannot Initialize FatFs! \n");
      }
      break;

    case HOST_USER_CONNECTION:
      App_State = APPLICATION_START;
      break;

    default:
      break;
  }
}

/**
  * @brief  System Clock Configuration
  * @param  None
  * @retval None
  */
static void APP_SystemClockConfig(void)
{
  RCC_OscInitTypeDef  OscInitstruct = {0};
  RCC_ClkInitTypeDef  ClkInitstruct = {0};
  
  OscInitstruct.OscillatorType  = RCC_OSCILLATORTYPE_HSE | RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_LSE | 
                                  RCC_OSCILLATORTYPE_LSI | RCC_OSCILLATORTYPE_HSI48M;
  OscInitstruct.HSEState        = RCC_HSE_OFF;                              /* Close HSE */
/* OscInitstruct.HSEFreq         = RCC_HSE_16_32MHz; */                       /* Choose HSE frequency of 16-32MHz */
  OscInitstruct.HSI48MState     = RCC_HSI48M_ON;                            /* Enable HSI48M */
  OscInitstruct.HSIState        = RCC_HSI_ON;                               /* Enable HSI */
  OscInitstruct.LSEState        = RCC_LSE_OFF;                              /* Close LSE */
/* OscInitstruct.LSEDriver       = RCC_LSEDRIVE_HIGH; */                    /* Drive capability level: high */
  OscInitstruct.LSIState        = RCC_LSI_OFF;                              /* Close LSI */
  OscInitstruct.PLL.PLLState    = RCC_PLL_ON;                               /* Enable PLL */
  OscInitstruct.PLL.PLLSource   = RCC_PLLSOURCE_HSI_DIV2;                   /* PLL clock source selection HSI/2 */
  OscInitstruct.PLL.PLLPrediv   = RCC_PLL_PREDIV_DIV1;                      /* PLL clock Prediv factor */
  OscInitstruct.PLL.PLLMUL      = 12;                                       /* PLL clock source 12-fold frequency */
  OscInitstruct.PLL.PLLPostdiv  = RCC_PLL_POSTDIV_DIV1;                     /* PLL clock Postdiv factor */
  /* Configure oscillator */
  if(HAL_RCC_OscConfig(&OscInitstruct) != HAL_OK)
  {
    APP_ErrorHandler();
  }
  
  ClkInitstruct.ClockType       = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2;
  ClkInitstruct.SYSCLKSource    = RCC_SYSCLKSOURCE_PLLCLK;              /* System clock selection PLLCLK */
  ClkInitstruct.AHBCLKDivider   = RCC_SYSCLK_DIV1;                      /* AHB clock 1 division */
  ClkInitstruct.APB1CLKDivider  = RCC_HCLK_DIV2;                        /* APB1 clock 2 division */
  ClkInitstruct.APB2CLKDivider  = RCC_HCLK_DIV2;                        /* APB2 clock 2 division */
  /* Configure Clock */
  if(HAL_RCC_ClockConfig(&ClkInitstruct, FLASH_LATENCY_4) != HAL_OK)
  {
    APP_ErrorHandler();
  }
}

/**
  * @brief  This function is executed in case of error occurrence.
  * @param  None
  * @retval None
  */
void APP_ErrorHandler(void)
{
  /* Infinite loop */
  while (1)
  {
  }
}

#ifdef  USE_FULL_ASSERT
/**
  * @brief  Reports the name of the source file and the source line number
  *         where the assert_param error has occurred.
  * @param  file: pointer to the source file name
  * @param  line: assert_param error line source number
  * @retval None
  */
void assert_failed(uint8_t *file, uint32_t line)
{
  /* User can add his own implementation to report the file name and line number,
     for example: printf("Wrong parameters value: file %s on line %d\r\n", file, line)  */
  /* Infinite loop */
  while (1)
  {
  }
}
#endif /* USE_FULL_ASSERT */

/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
