/**
  ******************************************************************************
  * @file    main.c
  * @author  MCU Application Team
  * @brief   Main program body
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) Puya Semiconductor Co.
  * All rights reserved.</center></h2>
  *
  * <h2><center>&copy; Copyright (c) 2016 STMicroelectronics.
  * All rights reserved.</center></h2>
  *
  * This software component is licensed by ST under BSD 3-Clause license,
  * the "License"; You may not use this file except in compliance with the
  * License. You may obtain a copy of the License at:
  *                        opensource.org/licenses/BSD-3-Clause
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "main.h"
#include "string.h"
#include "bootloader_host.h"

/* Private define ------------------------------------------------------------*/
#define USE_UART_MASTER
//#define USE_SPI_MASTER
//#define USE_I2C_MASTER

#define I2C_ADDRESS             (0xA6)

#define CODE_START_ADDR         (0x08004000)
#define CODE_SIZE               (0x00010000)
#define CODE_END_ADDR           (CODE_START_ADDR + CODE_SIZE)
#define CODE_BUFFER             ((uint8_t*)CODE_START_ADDR)

#define TARGET_FLASH_ADDR       (0x08000000)
#define TARGET_FLASH_SIZE       (0x00020000)
#define TARGET_PAGE_SIZE        (0x00000200)
#define TARGET_SECTOR_SIZE      (0x00001000)
#define TARGET_CODE_START       (0x08000000)


static void APP_SystemClockConfig(void);

uint8_t guc_DataBuffer[2 * TARGET_PAGE_SIZE];

static BL_MCU_Info_TypeDef mcu_py32e407 =
{
  .flash_addr  = TARGET_FLASH_ADDR,
  .flash_size  = TARGET_FLASH_SIZE,
  .page_size   = TARGET_PAGE_SIZE,
  .sector_size = TARGET_SECTOR_SIZE,
};

#ifdef USE_SPI_MASTER
static void APP_SPI_Init(void);
static int APP_SPI_ReadWrite(uint8_t *txdata, uint8_t *rxdata, int size, uint32_t timeout);

SPI_HandleTypeDef Spi1Handle;

static BL_IO_Interface_TypeDef bl_io =
{
  .IOType = BL_IO_SPI,
  .Init = NULL,
  .Deinit = NULL,
  .ReadWrite = APP_SPI_ReadWrite,
  .DefaultTimeout = 5000,
  .Buffer = guc_DataBuffer,
  .BufferSize = sizeof(guc_DataBuffer),
};
#endif

#ifdef USE_I2C_MASTER
static void APP_I2C_Init(void);
static int APP_I2C_ReadWrite(uint8_t *txdata, uint8_t *rxdata, int size, uint32_t timeout);

I2C_HandleTypeDef I2cHandle;

static BL_IO_Interface_TypeDef bl_io =
{
  .IOType = BL_IO_I2C,
  .Init = NULL,
  .Deinit = NULL,
  .ReadWrite = APP_I2C_ReadWrite,
  .DefaultTimeout = 5000,
  .Buffer = guc_DataBuffer,
  .BufferSize = sizeof(guc_DataBuffer),
};
#endif

#ifdef USE_UART_MASTER
static void APP_UART_Init(void);
static int APP_UART_ReadWrite(uint8_t *txdata, uint8_t *rxdata, int size, uint32_t timeout);

UART_HandleTypeDef UartHandle;

static BL_IO_Interface_TypeDef bl_io =
{
  .IOType = BL_IO_UART,
  .Init = NULL,
  .Deinit = NULL,
  .ReadWrite = APP_UART_ReadWrite,
  .DefaultTimeout = 5000,
  .Buffer = guc_DataBuffer,
  .BufferSize = sizeof(guc_DataBuffer),
};
#endif

/**
  * @brief  应用程序入口函数.
  * @retval int
  */
int main(void)
{
  /* 初始化所有外设，Flash接口，SysTick */
  HAL_Init();

  /* 系统时钟配置 */
  APP_SystemClockConfig();

  /* Log打印 */
  BSP_USART_Config();

  /* 初始化LED */
  BSP_LED_Init(LED_GREEN);

  /* 初始化LED */
  BSP_LED_Init(LED_GREEN);

  /* 初始化按键BUTTON */
  BSP_PB_Init(BUTTON_KEY, BUTTON_MODE_GPIO);

  #ifdef USE_SPI_MASTER
  /* 初始化SPI */
  APP_SPI_Init();
  #endif

  #ifdef USE_I2C_MASTER
  /* 初始化I2C */
  APP_I2C_Init();
  #endif

  #ifdef USE_UART_MASTER
  /* 初始化SPI */
  APP_UART_Init();
  #endif

  // 注册接口
  if (BootLoader_Register(&bl_io, &mcu_py32e407) < 0)
  {
    while(1);  //参数错误
  }

  /* 无限循环 */
  while (1)
  {
    /* 等待按键按下 */
    while (BSP_PB_GetState(BUTTON_USER))
    {
    }

    BootLoader_UpdateFlash(TARGET_CODE_START, CODE_BUFFER, CODE_SIZE, true);
  }
}

/**
  * @brief  应用程序入口函数.
  * @param  命令代码
  * @retval 无
  */
void BootLoader_CommandExecutingCallback(uint8_t cmd)
{
  BSP_LED_Toggle(LED3);
}

/**
  * @brief  应用程序入口函数.
  * @param  无
  * @retval 无
  */
void BootLoader_UpdateSuccessCallback(void)
{
  BSP_LED_On(LED3);
}

/**
  * @brief  应用程序入口函数.
  * @param  无
  * @retval 无
  */
void BootLoader_UpdateFailCallback(void)
{
  BSP_LED_Off(LED3);
}

/**
  * @brief  系统时钟配置函数
  * @param  无
  * @retval 无
  */
static void APP_SystemClockConfig(void)
{
  RCC_OscInitTypeDef  OscInitstruct = {0};
  RCC_ClkInitTypeDef  ClkInitstruct = {0};

  OscInitstruct.OscillatorType  = RCC_OSCILLATORTYPE_HSE | RCC_OSCILLATORTYPE_HSI | RCC_OSCILLATORTYPE_LSE |
                                  RCC_OSCILLATORTYPE_LSI | RCC_OSCILLATORTYPE_HSI48M;
  OscInitstruct.HSEState        = RCC_HSE_ON;                              /* 关闭HSE */
  OscInitstruct.HSEFreq         = RCC_HSE_16_32MHz;                      /* 选择HSE频率16~32M */
  OscInitstruct.HSI48MState     = RCC_HSI48M_OFF;                           /* 关闭HSI48M */
  /* OscInitstruct.HSI48MCalibrationValue = RCC_HSI48MCALIBRATION_DEFAULT; */ /* HSI48M默认校准值 */
  OscInitstruct.HSIState        = RCC_HSI_ON;                               /* 开启HSI */
  OscInitstruct.HSICalibrationValue = RCC_HSICALIBRATION_DEFAULT;           /* HSI默认校准值 */
  OscInitstruct.LSEState        = RCC_LSE_OFF;                              /* 关闭LSE */
  /* OscInitstruct.LSEDriver       = RCC_LSEDRIVE_HIGH; */                    /* 驱动能力等级：高 */
  OscInitstruct.LSIState        = RCC_LSI_OFF;                              /* 关闭LSI */
  OscInitstruct.PLL.PLLState    = RCC_PLL_OFF;                              /* 关闭PLL */
  OscInitstruct.PLL.PLLSource   = RCC_PLLSOURCE_HSE;                    /* PLL时钟源选择HSE */

  /*  OscInitstruct.PLL.PLLMUL      = RCC_PLL_MUL6; */                        /* PLL时钟源6倍频 */
  /* 配置振荡器 */
  if(HAL_RCC_OscConfig(&OscInitstruct) != HAL_OK)
  {
    while(1);
  }

  ClkInitstruct.ClockType       = RCC_CLOCKTYPE_SYSCLK | RCC_CLOCKTYPE_HCLK | RCC_CLOCKTYPE_PCLK1 | RCC_CLOCKTYPE_PCLK2;
  ClkInitstruct.SYSCLKSource    = RCC_SYSCLKSOURCE_HSE;                 /* 系统时钟选择HSI */
  ClkInitstruct.AHBCLKDivider   = RCC_SYSCLK_DIV1;                      /* AHB时钟1分频 */
  ClkInitstruct.APB1CLKDivider  = RCC_HCLK_DIV1;                        /* APB1时钟1分频 */
  ClkInitstruct.APB2CLKDivider  = RCC_HCLK_DIV1;                        /* APB2时钟2分频 */

  /* 配置时钟 */
  if(HAL_RCC_ClockConfig(&ClkInitstruct, FLASH_LATENCY_0) != HAL_OK)
  {
    while(1);
  }
}
#ifdef USE_SPI_MASTER
static void APP_SPI_Init(void)
{
  /*De-Initialize the SPI peripheral*/
  Spi1Handle.Instance               = SPI1;                       /* SPI1 */
  Spi1Handle.Init.BaudRatePrescaler = SPI_BAUDRATEPRESCALER_2;  /* prescaler :256 */
  Spi1Handle.Init.Direction         = SPI_DIRECTION_2LINES;       /* full duplex */
  Spi1Handle.Init.CLKPolarity       = SPI_POLARITY_LOW;           /* SPI Clock Polarity: low */
  Spi1Handle.Init.CLKPhase          = SPI_PHASE_1EDGE ;           /* Data sampling starts at the first clock edge */
  Spi1Handle.Init.DataSize          = SPI_DATASIZE_8BIT;          /* SPI Data Size is 8 bit */
  Spi1Handle.Init.FirstBit          = SPI_FIRSTBIT_MSB;           /* SPI MSB Transmission */
  Spi1Handle.Init.NSS               = SPI_NSS_HARD_OUTPUT;        /* NSS Hardware mode */
  Spi1Handle.Init.Mode = SPI_MODE_MASTER;                         /* Configure as host */
  Spi1Handle.Init.CRCCalculation = SPI_CRCCALCULATION_DISABLE;    /* The CRC check is disabled */

  /* Spi1Handle.Init.CRCPolynomial = 1; */                        /* CRC polynomial value */
  if (HAL_SPI_DeInit(&Spi1Handle) != HAL_OK)
  {
    while(1);
  }

  /* Initialize SPI peripheral */
  if (HAL_SPI_Init(&Spi1Handle) != HAL_OK)
  {
    while(1);
  }

  __HAL_SPI_ENABLE(&Spi1Handle);
}

static int APP_SPI_ReadWrite(uint8_t *txdata, uint8_t *rxdata, int size, uint32_t timeout)
{
  uint8_t tmp[TARGET_PAGE_SIZE + 0x10];

  if(txdata == NULL)
  {
    memset(tmp, 0xAA, size);
    txdata = tmp;
  }

  if(rxdata == NULL)
  {
    memset(tmp, 0x55, size);
    rxdata = tmp;
  }

  if(HAL_SPI_TransmitReceive(&Spi1Handle, txdata, rxdata, size, timeout) != HAL_OK)
  {
    return -1;
  }

  /* 等待从机执行命令完成 */
  HAL_Delay(1);

  return size;
}
#endif

#ifdef USE_I2C_MASTER
static void APP_I2C_Init(void)
{
  /* I2C初始化 */
  I2cHandle.Instance             = I2C1;                      /* I2C */
  I2cHandle.Init.ClockSpeed      = 100000;                    /* I2C通讯速度 */
  I2cHandle.Init.DutyCycle       = I2C_DUTYCYCLE_16_9;        /* I2C占空比 */
  I2cHandle.Init.OwnAddress1     = 0;                         /* I2C地址 */
  I2cHandle.Init.AddressingMode  = I2C_ADDRESSINGMODE_7BIT;   /* 7位寻址模式 */
  I2cHandle.Init.DualAddressMode = I2C_DUALADDRESS_DISABLE;   /* 不开启双地址 */
  /* I2cHandle.Init.OwnAddress2     = 0; */                   /* 第二个地址 */
  I2cHandle.Init.GeneralCallMode = I2C_GENERALCALL_DISABLE;   /* 禁止广播呼叫 */
  I2cHandle.Init.NoStretchMode   = I2C_NOSTRETCH_ENABLE;     /* 允许时钟延长 */

  if (HAL_I2C_Init(&I2cHandle) != HAL_OK)
  {
    while(1);
  }
}

static int APP_I2C_ReadWrite(uint8_t *txdata, uint8_t *rxdata, int size, uint32_t timeout)
{
  if(txdata == NULL)
  {
    if(HAL_I2C_Master_Receive(&I2cHandle, I2C_ADDRESS, rxdata, size, timeout) == HAL_OK)
    {
      return size;
    }

    /* 若从机NACK 1ms后重试 */
    HAL_Delay(1);
    return 0;
  }

  if(rxdata == NULL)
  {
    {
      if (HAL_I2C_Master_Transmit(&I2cHandle, I2C_ADDRESS, txdata, size, timeout) == HAL_OK)
      {
        return size;
      }

      /* 若从机NACK 1ms后重试 */
      HAL_Delay(1);
      return 0;
    }
  }

  return -1;
}
#endif

#ifdef USE_UART_MASTER
static void APP_UART_Init(void)
{
  /* Initialize USART */
  UartHandle.Instance          = USART1;
  UartHandle.Init.BaudRate     = 115200;
  UartHandle.Init.WordLength   = UART_WORDLENGTH_9B;
  UartHandle.Init.StopBits     = UART_STOPBITS_1;
  UartHandle.Init.Parity       = UART_PARITY_EVEN;
  UartHandle.Init.HwFlowCtl    = UART_HWCONTROL_NONE;
  UartHandle.Init.Mode         = UART_MODE_TX_RX;
  UartHandle.Init.OverSampling = UART_OVERSAMPLING_16;
  UartHandle.AdvancedInit.AdvFeatureInit = UART_ADVFEATURE_NO_INIT;
  HAL_UART_Init(&UartHandle);
}

static int APP_UART_ReadWrite(uint8_t *txdata, uint8_t *rxdata, int size, uint32_t timeout)
{
  if(txdata == NULL)
  {
    if(HAL_UART_Receive(&UartHandle, rxdata, size, timeout) == HAL_OK)
    {
      return size;
    }

    return -1;
  }

  if(rxdata == NULL)
  {
    if (HAL_UART_Transmit(&UartHandle, txdata, size, timeout) == HAL_OK)
    {
      return size;
    }

    return -1;
  }

  return -1;
}
#endif
/************************ (C) COPYRIGHT Puya *****END OF FILE******************/
