/**
  ******************************************************************************
  * @file    app_bootloader_host.c
  * @author  MCU Application Team
  * @brief   BootLoader application entry point
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2021 Puya Semiconductor.
  * All rights reserved.</center></h2>
  *
  *
  ******************************************************************************
  */

/* Includes ------------------------------------------------------------------*/
#include "bootloader_host.h"

/* Private types ------------------------------------------------------------*/
struct BootLoader_Host
{
  BL_MCU_Info_TypeDef *mcu;
  BL_IO_Interface_TypeDef *io;
  uint32_t CurrentTimeout;
};

/* Private macro -------------------------------------------------------------*/
#define SINGLE_ERASE_COUNT   0x10        //Single command to erase the number of sectors/pages
#define SINGLE_READ_COUNT    0x100       //Single command to read memory (MAX: 0x100)
/* Private variables ---------------------------------------------------------*/
static struct BootLoader_Host bl_host;

/* Private functions --------------------------------------------------------*/
static inline uint32_t BootLoader_AlignDown(uint32_t a, uint32_t align)
{
  return a - a % align;
}

static inline uint32_t BootLoader_AlignUp(uint32_t a, uint32_t align)
{
  return BootLoader_AlignDown(a + align - 1, align);
}

static uint8_t BootLoader_GetXOR(uint8_t *data, int size, uint8_t base)
{
  for (int i = 0; i < size; i++)
  {
    base ^= data[i];
  }

  return base;
}

static bool BootLoader_IsBlank(uint8_t *data, int size)
{
  for (uint32_t i = 0; i < size; i++)
  {
    if (data[i] != 0xFF)
    {
      return false;
    }
  }

  return true;
}

static void BootLoader_SetTimeOut(uint32_t timeout)
{
  bl_host.CurrentTimeout = timeout;
}

static int BootLoader_IO_Init(void)
{
  if(NULL != bl_host.io->Init)
  {
    if(bl_host.io->Init())
    {
      return BL_ERROR_IO_INIT;
    }
  }

  return BL_ERROR_OK;
}

static int BootLoader_IO_Deinit(void)
{
  if(NULL != bl_host.io->Deinit)
  {
    if(bl_host.io->Deinit())
    {
      return BL_ERROR_IO_DEINIT;
    }
  }

  return BL_ERROR_OK;
}

static int BootLoader_IO_Write(uint8_t *p_TxData, int size)
{
  int tmp = 0;
  int writesize = size;

  while(writesize > 0)
  {
    tmp = bl_host.io->ReadWrite(p_TxData, NULL, writesize, bl_host.CurrentTimeout);

    if(tmp < 0)
    {
      return BL_ERROR_IO_WRITE;
    }

    p_TxData += tmp;
    writesize -= tmp;
  }

  return BL_ERROR_OK;
}

static int BootLoader_IO_Read(uint8_t *p_RxData, int size)
{
  int tmp = 0;
  int readsize = size;

  while(readsize > 0)
  {
    tmp = bl_host.io->ReadWrite(NULL, p_RxData, readsize, bl_host.CurrentTimeout);

    if(tmp < 0)
    {
      return BL_ERROR_IO_READ;
    }

    p_RxData += tmp;
    readsize -= tmp;
  }

  return BL_ERROR_OK;
}

static int BootLoader_IO_ReadWrite(uint8_t *p_TxData, uint8_t *p_RxData, int size)
{
  int tmp = 0;
  int transfersize = size;

  while(transfersize > 0)
  {
    tmp = bl_host.io->ReadWrite(p_TxData, p_RxData, transfersize, bl_host.CurrentTimeout);

    if(tmp < 0)
    {
      return BL_ERROR_IO_READ;
    }

    p_TxData += tmp;
    p_RxData += tmp;
    transfersize -= tmp;
  }

  return BL_ERROR_OK;
}

static int BootLoader_IO_WriteByte(uint8_t ucData)
{
  return BootLoader_IO_Write(&ucData, 1);
}

static int BootLoader_WaitACK(void)
{
  uint8_t ucData;

  if (BootLoader_IO_Read(&ucData, 1) != BL_ERROR_OK)
  {
    return BL_ERROR_IO_READ;
  }

  if (ACK_BYTE != ucData)
  {
    return BL_ERROR_ACK;
  }

  return BL_ERROR_OK;
}

static int BootLoader_HandShake(uint32_t trytimes, uint32_t timeout)
{
  uint8_t tmp_tx = 0x7F;
  uint8_t tmp_rx = 0x0;

  BootLoader_SetTimeOut(timeout);

  for (int i = 0; i < trytimes; i++)
  {
    switch(bl_host.io->IOType)
    {
      case BL_IO_SPI:
        if(BootLoader_IO_ReadWrite(&tmp_tx, &tmp_rx, 1) != BL_ERROR_OK)
        {
          continue;
        }
        break;

      default:
        if(BootLoader_IO_WriteByte(0x7F) != BL_ERROR_OK)
        {
          return BL_ERROR_IO_WRITE;
        }

        if (BootLoader_IO_Read(&tmp_rx, 1) != BL_ERROR_OK)
        {
          continue;
        }
        break;
    }

    if ((ACK_BYTE == tmp_rx || NACK_BYTE == tmp_rx))
    {
      TRACE("Established connection after %d times\r\n", i);
      BootLoader_SetTimeOut(bl_host.io->DefaultTimeout);
      return BL_ERROR_OK;
    }
  }

  return BL_ERROR_ACK;
}

static int BootLoader_SendByteAndNByte(uint8_t data)
{
  uint8_t tmparray[2];
  tmparray[0] = data;
  tmparray[1] = ~data;

  if(BootLoader_IO_Write(tmparray, 2) < 0)
  {
    return BL_ERROR_IO_WRITE;
  }

  return BootLoader_WaitACK();
}

static int BootLoader_SendAddress(uint32_t addr)
{
  uint8_t tmparray[5];
  tmparray[0] = HIBYTE(HIWORD(addr));
  tmparray[1] = LOBYTE(HIWORD(addr));
  tmparray[2] = HIBYTE(LOWORD(addr));
  tmparray[3] = LOBYTE(LOWORD(addr));
  tmparray[4] = BootLoader_GetXOR(tmparray, 4, 0);
  BootLoader_IO_Write(tmparray, 5);
  return BootLoader_WaitACK();
}

/* BootLoader command functions --------------------------------------------------------*/
static int CMD_ExtendErase(uint8_t *pucPageindex, uint32_t size, BL_Erase_Typedef erasetype)
{
  int status;
  uint8_t erase;

  BootLoader_CommandExecutingCallback(CMD_EXT_ERASE_MEMORY);

  status = BootLoader_SendByteAndNByte(CMD_EXT_ERASE_MEMORY);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  switch(erasetype)
  {
    case BL_ERASE_PAGE:
      erase = 0x10;
      break;

    case BL_ERASE_SECTOR:
      erase = 0x20;
      break;

    default:
      TRACE("Unsupported erasing type\r\n");
      return BL_ERROR_ARGUMENT;
  }

  status = BootLoader_IO_WriteByte(erase);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_WriteByte(size / 2 - 1);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_Write(pucPageindex, size);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_WriteByte(BootLoader_GetXOR(pucPageindex, size, 0) ^ erase ^ (size / 2 - 1));

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  if(bl_host.io->IOType == BL_IO_SPI)
  {
    HAL_Delay(100);
  }

  return BootLoader_WaitACK();
}

static int CMD_ExtWriteMemory(uint32_t addr, uint8_t *pucDataBuf, int size)
{
  int status;
  uint8_t tmp[2];
  uint8_t xor;

  BootLoader_CommandExecutingCallback(CMD_EXT_WRITE_MEMORY);

  status = BootLoader_SendByteAndNByte(CMD_EXT_WRITE_MEMORY);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_SendAddress(addr);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  tmp[0] = HIBYTE(size - 1);
  tmp[1] = LOBYTE(size - 1);
  xor = tmp[0] ^ tmp[1];

  status = BootLoader_IO_Write(tmp, 2);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_Write(pucDataBuf, size);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_WriteByte(BootLoader_GetXOR(pucDataBuf, size, xor));

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  return BootLoader_WaitACK();
}

static int CMD_WriteMemory(uint32_t addr, uint8_t *pucDataBuf, int size)
{
  int status;

  BootLoader_CommandExecutingCallback(CMD_WRITE_MEMORY);

  if(size > 256)
  {
    return BL_ERROR_ARGUMENT;
  }

  status = BootLoader_SendByteAndNByte(CMD_WRITE_MEMORY);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_SendAddress(addr);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_WriteByte(size - 1);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_Write(pucDataBuf, size);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_WriteByte(BootLoader_GetXOR(pucDataBuf, size, size - 1));

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  return BootLoader_WaitACK();
}

static int CMD_ReadMemory(uint32_t addr, uint8_t *pucDataBuf, int size)
{
  int status;
  uint8_t tmp[2];

  BootLoader_CommandExecutingCallback(CMD_READ_MEMORY);

  status = BootLoader_SendByteAndNByte(CMD_READ_MEMORY);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_SendAddress(addr);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  tmp[0] = size - 1;
  tmp[1] = ~tmp[0];

  status = BootLoader_IO_Write(tmp, 2);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_WaitACK();

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_IO_Read(pucDataBuf, size);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  return BL_ERROR_OK;
}

static int CMD_Go(uint32_t addr)
{
  int status;

  BootLoader_CommandExecutingCallback(CMD_GO);

  status = BootLoader_SendByteAndNByte(CMD_GO);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  status = BootLoader_SendAddress(addr);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  return BL_ERROR_OK;
}

/* Exported functions --------------------------------------------------------*/
int BootLoader_Register(BL_IO_Interface_TypeDef *io, BL_MCU_Info_TypeDef *mcu)
{
  if(NULL == io || NULL == mcu)
  {
    return BL_ERROR_ARGUMENT;
  }

  if(NULL == io->ReadWrite)
  {
    return BL_ERROR_ARGUMENT;
  }

  if(io->BufferSize < 2 * mcu->page_size)
  {
    return BL_ERROR_ARGUMENT;
  }

  bl_host.io = io;
  bl_host.mcu = mcu;

  return BL_ERROR_OK;
}

int BootLoader_Connect(uint32_t trytimes, uint32_t timeout)
{
  int status;

  status = BootLoader_IO_Init();

  if(status < 0)
  {
    return status;
  }

  status = BootLoader_HandShake(trytimes, timeout);

  if(status < 0)
  {
    return status;
  }

  return BL_ERROR_OK;
}

int BootLoader_Disconnect(void)
{
  return BootLoader_IO_Deinit();
}

int BootLoader_Erase(uint32_t startaddr, int size, BL_Erase_Typedef erasetype)
{
  int status;
  uint8_t buffer[SINGLE_ERASE_COUNT * 2];
  uint32_t start;
  uint32_t end;
  uint32_t erasesize;

  if (size <= 0)
  {
    return BL_ERROR_ARGUMENT;
  }

  switch(erasetype)
  {
    case BL_ERASE_PAGE:
      erasesize = bl_host.mcu->page_size;
      break;

    default:
      erasesize = bl_host.mcu->sector_size;
      break;
  }

  start = BootLoader_AlignDown(startaddr, erasesize);
  end = BootLoader_AlignUp(startaddr + size, erasesize);

  if (start < bl_host.mcu->flash_addr || end > bl_host.mcu->flash_addr + bl_host.mcu->flash_size)
  {
    return BL_ERROR_ARGUMENT;
  }

  start -= bl_host.mcu->flash_addr;
  end -= bl_host.mcu->flash_addr;

  start /= erasesize;
  end /= erasesize;

  while (start < end)
  {
    uint32_t tmp = ((end - start) > SINGLE_ERASE_COUNT) ? SINGLE_ERASE_COUNT : end - start;

    for (uint32_t i = 0; i < tmp; i++)
    {
      buffer[2 * i] = HIBYTE(i + start);
      buffer[2 * i + 1] = LOBYTE(i + start);
    }

    status = CMD_ExtendErase(buffer, 2 * tmp, erasetype);

    if (BL_ERROR_OK != status)
    {
      return status;
    }

    TRACE("Erase: 0x%X@0x%08X\r\n", tmp * erasesize, bl_host.mcu->flash_addr + start * erasesize);

    start += tmp;
  }

  return BL_ERROR_OK;
}

int BootLoader_Program(uint32_t addr, uint8_t *pucDataBuf, int size)
{
  int status;
  uint32_t tmp;
  uint32_t count = 0;

  while (count < size)
  {
    tmp = ((size - count) > bl_host.mcu->page_size) ? bl_host.mcu->page_size : size - count;

    if (!BootLoader_IsBlank(pucDataBuf + count, tmp))
    {
      if(bl_host.mcu->page_size > 0x100)
      {
        status = CMD_ExtWriteMemory(addr + count, pucDataBuf + count, tmp);

        if (BL_ERROR_OK != status)
        {
          return status;
        }
      }
      else
      {
        status = CMD_WriteMemory(addr + count, pucDataBuf + count, tmp);

        if (BL_ERROR_OK != status)
        {
          return status;
        }
      }

      TRACE("Program: 0x%X@0x%08X\r\n", tmp, addr + count);
    }

    count += tmp;
  }

  return BL_ERROR_OK;
}

int BootLoader_Read(uint32_t addr, uint8_t *pucDataBuf, int size)
{
  int status;
  const uint32_t Single_Read_Count = 0x100; //Max:0x100
  int count = 0;
  int tmp;

  while (count < size)
  {
    tmp = ((size - count) > Single_Read_Count) ? Single_Read_Count : size - count;

    status = CMD_ReadMemory(addr + count, pucDataBuf + count, tmp);

    if (BL_ERROR_OK != status)
    {
      return status;
    }

    TRACE("Read: 0x%X@0x%08X\r\n", tmp, addr + count);

    count += tmp;
  }

  return BL_ERROR_OK;
}

int BootLoader_GoTo(uint32_t addr)
{
  int status;

  status = CMD_Go(addr);

  if (BL_ERROR_OK != status)
  {
    return status;
  }

  TRACE("GoTo: 0x%08X\r\n", addr);

  return BL_ERROR_OK;
}

int BootLoader_Verify(uint32_t addr, uint8_t *data, int size)
{
  int status;
  int tmpsize;
  int verifysize;
  uint32_t count = 0;
  uint32_t verifyaddr;
  uint8_t *readbuf;

  status = 0;
  verifysize = size;
  verifyaddr = addr;
  readbuf = bl_host.io->Buffer;

  while(verifysize > 0)
  {
    tmpsize = MIN(verifysize, 0x100);
    tmpsize = MIN(tmpsize, bl_host.mcu->page_size);

    if (!BootLoader_IsBlank(data + count, tmpsize))
    {
      status = BootLoader_Read(verifyaddr, readbuf, tmpsize);

      if(status < 0)
      {
        goto End;
      }

      for (int i = 0; i < tmpsize; i++)
      {
        if (readbuf[i] != data[count + i])
        {
          TRACE("Verify fail at: 0x%08X\r\n", verifyaddr + i);
          status = BL_ERROR;
          goto End;
        }
      }

      TRACE("Verify: 0x%X@0x%08X\r\n", tmpsize, verifyaddr);
    }

    count += tmpsize;
    verifyaddr += tmpsize;
    verifysize -= tmpsize;
  }

End:
  return status;
}

__weak void BootLoader_CommandExecutingCallback(uint8_t cmd)
{
  (void)cmd;
}

/**
  * @brief  Flash
  * @param  
  * @retval 
  */
int BootLoader_UpdateFlash(uint32_t addr, uint8_t *data, int size, bool jump)
{
  int errorcode;
  TRACE("Update Start\r\n");

  // 
  errorcode = BootLoader_Connect(10, 100);

  if (errorcode < 0)
  {
    TRACE("Connect FAIL\r\n");
    goto End;
  }

  // 
  errorcode = BootLoader_Erase(addr, size, BL_ERASE_SECTOR);

  if (errorcode < 0)
  {
    TRACE("Erase FAIL\r\n");
    goto End;
  }

  // д
  errorcode = BootLoader_Program(addr, data, size);

  if (errorcode < 0)
  {
    TRACE("Program FAIL\r\n");
    goto End;
  }

  // У
  errorcode = BootLoader_Verify(addr, data, size);

  if (errorcode < 0)
  {
    TRACE("Verify FAIL\r\n");
    goto End;
  }

  // ת
  if(jump)
  {
    errorcode = BootLoader_GoTo(addr);

    if (errorcode < 0)
    {
      goto End;
    }
  }

End:

  if(errorcode < 0)
  {
    TRACE("Update FAIL\r\n");
    BootLoader_UpdateFailCallback();
  }
  else
  {
    TRACE("Update SUCCESS\r\n");
    BootLoader_UpdateSuccessCallback();
  }

  BootLoader_Disconnect();
  return errorcode;
}

__weak void BootLoader_UpdateSuccessCallback(void)
{

}

__weak void BootLoader_UpdateFailCallback(void)
{

}

/************************ (C) COPYRIGHT Puya *****END OF FILE****/
