/**
  ******************************************************************************
  * @file    bootloader_host.h
  * @author  MCU Application Team
  * @brief   Header for app_bootloader.c module
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; Copyright (c) 2020 Puya Semiconductor.
  * All rights reserved.</center></h2>
  *
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __BOOTLOADER_H
#define __BOOTLOADER_H

/* Includes ------------------------------------------------------------------*/
#include <stdio.h>
#include <stdint.h>
#include <stdbool.h>
#include <stddef.h>
#include "main.h"

/* Exported types ------------------------------------------------------------*/
typedef enum
{
  BL_ERROR_OK         = 0,
  BL_ERROR            = -1,
  BL_ERROR_ARGUMENT   = -2,
  BL_ERROR_IO_INIT    = -3,
  BL_ERROR_IO_DEINIT  = -4,
  BL_ERROR_IO_READ    = -5,
  BL_ERROR_IO_WRITE   = -6,
  BL_ERROR_ACK        = -7,
} BL_Status;

typedef enum
{
  BL_IO_UART,
  BL_IO_SPI,
  BL_IO_I2C,
} BL_IO_Typedef;

typedef enum
{
  BL_ERASE_PAGE,
  BL_ERASE_SECTOR,
} BL_Erase_Typedef;

typedef struct
{
  uint32_t flash_size;
  uint32_t flash_addr;
  uint32_t page_size;
  uint32_t sector_size;
} BL_MCU_Info_TypeDef;

typedef struct
{
  BL_IO_Typedef IOType;
  int (*Init)(void);
  int (*Deinit)(void);
  int (*ReadWrite)(uint8_t *p_TxData, uint8_t *p_RxData, int size, uint32_t timeout);
  uint32_t DefaultTimeout;
  uint8_t * Buffer;
  uint32_t BufferSize;
} BL_IO_Interface_TypeDef;

/* Exported constants --------------------------------------------------------*/
/* Exported macro ------------------------------------------------------------*/

#define LOG_TRACE

#ifdef LOG_TRACE
  #define TRACE   printf
#else
  #define TRACE(...)
#endif


#define ERROR_COMMAND                     0xEC             /* Error command */
#define ACK_BYTE                          0x79             /* Acknowledge Byte ID */
#define NACK_BYTE                         0x1F             /* No Acknowledge Byte ID */
#define BUSY_BYTE                         0x76             /* Busy Byte */
#define SYNC_BYTE                         0xA5             /* synchronization byte */

#define CMD_GET_COMMAND                   0x00             /* Get commands command */
#define CMD_GET_VERSION                   0x01             /* Get Version command */
#define CMD_GET_ID                        0x02             /* Get ID command */
#define CMD_GET_DEVICE_IDCODE             0x03             /* Get DEVICE_IDCODE command */
#define CMD_READ_MEMORY                   0x11             /* Read Memory command */
#define CMD_WRITE_MEMORY                  0x31             /* Write Memory command */
#define CMD_EXT_WRITE_MEMORY              0x32             /* Extend Write Memory command */
#define CMD_GO                            0x21             /* GO command */
#define CMD_READ_PROTECT                  0x82             /* Readout Protect command */
#define CMD_READ_UNPROTECT                0x92             /* Readout Unprotect command */
#define CMD_EXT_ERASE_MEMORY              0x44             /* Erase Memory command */
#define CMD_WRITE_PROTECT                 0x63             /* Write Protect command */
#define CMD_WRITE_UNPROTECT               0x73             /* Write? ??Unprotect command */

#define HIWORD(x) (((x)>>16)&0xFFFF)
#define LOWORD(x) ((x)&0xFFFF)

#define HIBYTE(x) (((x)>>8)&0xFF)
#define LOBYTE(x) ((x)&0xFF)

#define MAX(a,b) (a>b?a:b)
#define MIN(a,b) (a<b?a:b)


/* Exported constants --------------------------------------------------------*/
/* Exported variable ------------------------------------------------------- */
/* Exported functions ------------------------------------------------------- */
int BootLoader_Register(BL_IO_Interface_TypeDef *io, BL_MCU_Info_TypeDef *mcu);
int BootLoader_Connect(uint32_t trytimes, uint32_t timeout);
int BootLoader_Disconnect(void);
int BootLoader_Erase(uint32_t startaddr, int size, BL_Erase_Typedef erasetype);
int BootLoader_Program(uint32_t addr, uint8_t *data, int size);
int BootLoader_Read(uint32_t addr, uint8_t *data, int size);
int BootLoader_GoTo(uint32_t addr);
int BootLoader_Verify(uint32_t addr, uint8_t *data, int size);
void BootLoader_CommandExecutingCallback(uint8_t cmd);

int BootLoader_UpdateFlash(uint32_t addr, uint8_t *data, int size, bool jump);
void BootLoader_UpdateSuccessCallback(void);
void BootLoader_UpdateFailCallback(void);

#endif /* __BOOTLOADER_H */

/************************ (C) COPYRIGHT Puya Semiconductor *****END OF FILE****/
